"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubminuteStateMachine = exports.IteratorLambda = exports.LambdaSubminute = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk = require("aws-cdk-lib");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_lambda_nodejs_1 = require("aws-cdk-lib/aws-lambda-nodejs");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
class LambdaSubminute extends constructs_1.Construct {
    constructor(parent, name, props) {
        super(parent, name);
        const iterator = new IteratorLambda(this, 'IteratorLambda', { targetFunction: props.targetFunction });
        this.iteratorFunction = iterator.function;
        const subminuteStateMachine = new SubminuteStateMachine(this, 'SubminuteStateMachine', {
            stateMachineName: 'lambda-subminute-statemachine',
            targetFunction: props.targetFunction,
            iteratorFunction: this.iteratorFunction,
            intervalTime: props.intervalTime ?? 10,
            frequency: props.frequency ?? 6,
        });
        this.stateMachineArn = subminuteStateMachine.stateMachine.stateMachineArn;
        const startRule = new events.Rule(this, 'StartSubminuteStateMachine', {
            schedule: events.Schedule.expression(props.cronjobExpression ?? 'cron(50/1 15-17 ? * * *)'),
            ruleName: 'subminute-statemachine-lambda-rule',
            description: `A rule to run the subminute state machine, i.e. ${subminuteStateMachine.stateMachine.stateMachineName}`,
        });
        startRule.addTarget(new targets.SfnStateMachine(subminuteStateMachine.stateMachine, {
            input: events.RuleTargetInput.fromObject({
                iterator: {
                    index: 0,
                    count: 6,
                },
            }),
        }));
    }
}
exports.LambdaSubminute = LambdaSubminute;
_a = JSII_RTTI_SYMBOL_1;
LambdaSubminute[_a] = { fqn: "cdk-lambda-subminute.LambdaSubminute", version: "2.0.26" };
class IteratorLambda extends constructs_1.Construct {
    constructor(scope, name, props) {
        super(scope, name);
        const iteratorLambdaRole = new iam.Role(this, 'IteratorLambdaRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('lambda.amazonaws.com')),
            description: 'An execution role for a Lambda function to invoke a target Lambda Function per time unit less than one minute',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AWSXRayDaemonWriteAccess'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'),
            ],
            roleName: 'Lambda-Iterator-Role',
        });
        cdk.DockerVolumeConsistency.CONSISTENT;
        iteratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            sid: 'TargetLambdaPermission',
            effect: iam.Effect.ALLOW,
            actions: ['lambda:InvokeFunction'],
            resources: [props.targetFunction.functionArn],
        }));
        this.function = new aws_lambda_nodejs_1.NodejsFunction(this, 'Iterator', {
            functionName: 'lambda-subminute-iterator',
            description: 'A function for breaking the limit of 1 minute with the CloudWatch Rules.',
            logRetention: aws_logs_1.RetentionDays.THREE_MONTHS,
            runtime: aws_lambda_1.Runtime.NODEJS_14_X,
            entry: fs.existsSync(path.join(__dirname, 'resources/iterator/iterator_agent.ts')) ? path.join(__dirname, 'resources/iterator/iterator_agent.ts') : path.join(__dirname, 'resources/iterator/iterator_agent.js'),
            handler: 'lambdaHandler',
            environment: {
                TARGET_FN_NAME: props.targetFunction.functionName,
            },
            memorySize: 128,
            role: iteratorLambdaRole,
            timeout: cdk.Duration.seconds(58),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
    }
}
exports.IteratorLambda = IteratorLambda;
_b = JSII_RTTI_SYMBOL_1;
IteratorLambda[_b] = { fqn: "cdk-lambda-subminute.IteratorLambda", version: "2.0.26" };
class SubminuteStateMachine extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Creates a state machine for breaking the limit of 1 minute with the CloudWatch Rules.
         *
         * @param iteratorFunction The iterator Lambda function for the target Labmda funciton.
         * @param intervalTime Seconds for an interval, the product of `frequency` and `intervalTime` should be approximagely 1 minute.
         * @param frequency How many times you intent to execute in a minute.
         * @returns THe job definition for the state machine.
         */
        this.createJobDefinition = (iteratorFunction, intervalTime, frequency) => {
            const configureCount = new sfn.Pass(this, 'ConfigureCount', {
                result: sfn.Result.fromObject({
                    index: 0,
                    count: frequency,
                }),
                resultPath: '$.iterator',
            });
            const iterator = new tasks.LambdaInvoke(this, 'Iterator', {
                lambdaFunction: iteratorFunction,
                resultPath: '$.iterator',
                resultSelector: {
                    'index.$': '$.Payload.index',
                    'count.$': '$.Payload.count',
                    'continue.$': '$.Payload.continue',
                },
            });
            const wait = new sfn.Wait(this, 'Wait for the target Lambda function finished', {
                time: sfn.WaitTime.duration(cdk.Duration.seconds(intervalTime)),
            });
            wait.next(iterator);
            const done = new sfn.Pass(this, 'Done');
            const isCountReached = new sfn.Choice(this, 'IsCountReached');
            isCountReached.when(sfn.Condition.booleanEquals('$.iterator.continue', true), wait);
            isCountReached.otherwise(done);
            const jobDefinition = configureCount.next(iterator).next(isCountReached);
            return jobDefinition;
        };
        /**
         * Creates a role and corresponding policies for the subminute state machine.
         *
         * @param targetFunctionArn the ARN of the Lambda function that executes your intention.
         * @param iteratorFunctionArn the ARN of the iterator Lambda function for the target Lambda function.
         * @returns the role as the documentation indicates.
         */
        this._createWorkFlowRole = (targetFunctionArn, iteratorFunctionArn) => {
            const workFlowExecutionRole = new iam.Role(this, 'StepFunctionExecutionRole', {
                assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
                description: 'Execute a workflow related to executing a Lambda function per time unit less than 1 minute.',
            });
            workFlowExecutionRole.addToPolicy(new iam.PolicyStatement({
                sid: 'LambdaInvokePermissions',
                effect: iam.Effect.ALLOW,
                actions: ['lambda:InvokeFunction'],
                resources: [
                    targetFunctionArn,
                    iteratorFunctionArn,
                ],
            }));
            return workFlowExecutionRole;
        };
        const stateMachineRole = this._createWorkFlowRole(props.targetFunction.functionArn, props.iteratorFunction.functionArn);
        const jobDefinition = this.createJobDefinition(props.iteratorFunction, props.intervalTime, props.frequency);
        const stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            stateMachineName: props.stateMachineName,
            definition: jobDefinition,
            role: stateMachineRole,
        });
        this.stateMachine = stateMachine;
    }
}
exports.SubminuteStateMachine = SubminuteStateMachine;
_c = JSII_RTTI_SYMBOL_1;
SubminuteStateMachine[_c] = { fqn: "cdk-lambda-subminute.SubminuteStateMachine", version: "2.0.26" };
//# sourceMappingURL=data:application/json;base64,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