"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseEdgeConstruct = void 0;
const core_1 = require("@aws-cdk/core");
class BaseEdgeConstruct extends core_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.edgeStack =
            this.stack.region !== 'us-east-1'
                ? this.getOrCreateCrossRegionSupportStack()
                : this.stack;
    }
    getOrCreateCrossRegionSupportStack() {
        const { account, stackName } = this.stack;
        const stackId = `lambda-at-edge-support-stack`;
        const app = this.requireApp();
        let supportStack = app.node.tryFindChild(stackId);
        if (!supportStack) {
            supportStack = new core_1.Stack(app, stackId, {
                stackName: `${stackName}-support-lambda-at-edge`,
                env: {
                    account,
                    region: 'us-east-1',
                },
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
            // the stack containing the edge lambdas must be deployed before
            this.stack.addDependency(supportStack);
        }
        return supportStack;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Stacks which uses edge constructs must be part of a CDK app');
        }
        return app;
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer
                    .cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
}
exports.BaseEdgeConstruct = BaseEdgeConstruct;
//# sourceMappingURL=data:application/json;base64,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