"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Builder
 */
class Builder {
    constructor(options) {
        this.options = options;
        // Original package.json
        const pkgPath = util_1.findPkgPath();
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.originalPkg = fs.readFileSync(this.pkgPath);
        this.originalPkgJson = JSON.parse(this.originalPkg.toString());
    }
    /**
     * Build with parcel in a Docker container
     */
    build() {
        var _a, _b, _c, _d;
        try {
            this.updatePkg();
            const dockerBuildArgs = [
                'build',
                '--build-arg', `NODE_TAG=${this.options.nodeDockerTag}`,
                '-t', 'parcel-bundler',
                path.join(__dirname, '../parcel-bundler'),
            ];
            const build = child_process_1.spawnSync('docker', dockerBuildArgs);
            if (build.error) {
                throw build.error;
            }
            if (build.status !== 0) {
                throw new Error(`[Status ${build.status}] stdout: ${(_a = build.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = build.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
            }
            const containerProjectRoot = '/project';
            const containerOutDir = '/out';
            const containerCacheDir = '/cache';
            const containerEntryPath = path.join(containerProjectRoot, path.relative(this.options.projectRoot, path.resolve(this.options.entry)));
            const dockerRunArgs = [
                'run', '--rm',
                '-v', `${this.options.projectRoot}:${containerProjectRoot}`,
                '-v', `${path.resolve(this.options.outDir)}:${containerOutDir}`,
                ...(this.options.cacheDir ? ['-v', `${path.resolve(this.options.cacheDir)}:${containerCacheDir}`] : []),
                ...flatten(Object.entries(this.options.environment || {}).map(([k, v]) => ['--env', `${k}=${v}`])),
                '-w', path.dirname(containerEntryPath).replace(/\\/g, '/'),
                'parcel-bundler',
            ];
            const parcelArgs = [
                'parcel', 'build', containerEntryPath.replace(/\\/g, '/'),
                '--out-dir', containerOutDir,
                '--out-file', 'index.js',
                '--global', this.options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !this.options.minify && '--no-minify',
                !this.options.sourceMaps && '--no-source-maps',
                ...(this.options.cacheDir ? ['--cache-dir', containerCacheDir] : []),
            ].filter(Boolean);
            const parcel = child_process_1.spawnSync('docker', [...dockerRunArgs, ...parcelArgs]);
            if (parcel.error) {
                throw parcel.error;
            }
            if (parcel.status !== 0) {
                throw new Error(`[Status ${parcel.status}] stdout: ${(_c = parcel.stdout) === null || _c === void 0 ? void 0 : _c.toString().trim()}\n\n\nstderr: ${(_d = parcel.stderr) === null || _d === void 0 ? void 0 : _d.toString().trim()}`);
            }
        }
        catch (err) {
            throw new Error(`Failed to build file at ${this.options.entry}: ${err}`);
        }
        finally { // Always restore package.json to original
            this.restorePkg();
        }
    }
    /**
     * Updates the package.json to configure Parcel
     */
    updatePkg() {
        const updateData = {};
        // Update engines.node (Babel target)
        updateData.engines = { node: `>= ${this.options.nodeVersion}` };
        // Write new package.json
        if (Object.keys(updateData).length !== 0) {
            fs.writeFileSync(this.pkgPath, JSON.stringify({
                ...this.originalPkgJson,
                ...updateData,
            }, null, 2));
        }
    }
    restorePkg() {
        fs.writeFileSync(this.pkgPath, this.originalPkg);
    }
}
exports.Builder = Builder;
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVpbGRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJ1aWxkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxpREFBMEM7QUFDMUMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QixpQ0FBcUM7QUE4RHJDOztHQUVHO0FBQ0gsTUFBYSxPQUFPO0lBT2xCLFlBQTZCLE9BQXVCO1FBQXZCLFlBQU8sR0FBUCxPQUFPLENBQWdCO1FBQ2xELHdCQUF3QjtRQUN4QixNQUFNLE9BQU8sR0FBRyxrQkFBVyxFQUFFLENBQUM7UUFDOUIsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUNELElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDbEQsSUFBSSxDQUFDLFdBQVcsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7T0FFRztJQUNJLEtBQUs7O1FBQ1YsSUFBSTtZQUNGLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUVqQixNQUFNLGVBQWUsR0FBRztnQkFDdEIsT0FBTztnQkFDUCxhQUFhLEVBQUUsWUFBWSxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFBRTtnQkFDdkQsSUFBSSxFQUFFLGdCQUFnQjtnQkFDdEIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsbUJBQW1CLENBQUM7YUFDMUMsQ0FBQztZQUVGLE1BQU0sS0FBSyxHQUFHLHlCQUFTLENBQUMsUUFBUSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBRW5ELElBQUksS0FBSyxDQUFDLEtBQUssRUFBRTtnQkFDZixNQUFNLEtBQUssQ0FBQyxLQUFLLENBQUM7YUFDbkI7WUFFRCxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLFdBQVcsS0FBSyxDQUFDLE1BQU0sYUFBYSxNQUFBLEtBQUssQ0FBQyxNQUFNLDBDQUFFLFFBQVEsR0FBRyxJQUFJLEVBQUUsaUJBQWlCLE1BQUEsS0FBSyxDQUFDLE1BQU0sMENBQUUsUUFBUSxHQUFHLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQzthQUN4STtZQUVELE1BQU0sb0JBQW9CLEdBQUcsVUFBVSxDQUFDO1lBQ3hDLE1BQU0sZUFBZSxHQUFHLE1BQU0sQ0FBQztZQUMvQixNQUFNLGlCQUFpQixHQUFHLFFBQVEsQ0FBQztZQUNuQyxNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRXRJLE1BQU0sYUFBYSxHQUFHO2dCQUNwQixLQUFLLEVBQUUsTUFBTTtnQkFDYixJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsSUFBSSxvQkFBb0IsRUFBRTtnQkFDM0QsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLGVBQWUsRUFBRTtnQkFDL0QsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsSUFBSSxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDdkcsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsSUFBSSxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUNsRyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDO2dCQUMxRCxnQkFBZ0I7YUFDakIsQ0FBQztZQUNGLE1BQU0sVUFBVSxHQUFHO2dCQUNqQixRQUFRLEVBQUUsT0FBTyxFQUFFLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDO2dCQUN6RCxXQUFXLEVBQUUsZUFBZTtnQkFDNUIsWUFBWSxFQUFFLFVBQVU7Z0JBQ3hCLFVBQVUsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU07Z0JBQy9CLFVBQVUsRUFBRSxNQUFNO2dCQUNsQix1QkFBdUI7Z0JBQ3ZCLGFBQWEsRUFBRSxHQUFHO2dCQUNsQixDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxJQUFJLGFBQWE7Z0JBQ3JDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLElBQUksa0JBQWtCO2dCQUM5QyxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxFQUFFLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUNyRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQWEsQ0FBQztZQUU5QixNQUFNLE1BQU0sR0FBRyx5QkFBUyxDQUFDLFFBQVEsRUFBRSxDQUFDLEdBQUcsYUFBYSxFQUFFLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUV0RSxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUU7Z0JBQ2hCLE1BQU0sTUFBTSxDQUFDLEtBQUssQ0FBQzthQUNwQjtZQUVELElBQUksTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7Z0JBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsV0FBVyxNQUFNLENBQUMsTUFBTSxhQUFhLE1BQUEsTUFBTSxDQUFDLE1BQU0sMENBQUUsUUFBUSxHQUFHLElBQUksRUFBRSxpQkFBaUIsTUFBQSxNQUFNLENBQUMsTUFBTSwwQ0FBRSxRQUFRLEdBQUcsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO2FBQzNJO1NBQ0Y7UUFBQyxPQUFPLEdBQUcsRUFBRTtZQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxLQUFLLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDMUU7Z0JBQVMsRUFBRSwwQ0FBMEM7WUFDcEQsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1NBQ25CO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ssU0FBUztRQUNmLE1BQU0sVUFBVSxHQUEyQixFQUFFLENBQUM7UUFDOUMscUNBQXFDO1FBQ3JDLFVBQVUsQ0FBQyxPQUFPLEdBQUcsRUFBRSxJQUFJLEVBQUUsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUM7UUFFaEUseUJBQXlCO1FBQ3pCLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hDLEVBQUUsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDO2dCQUM1QyxHQUFHLElBQUksQ0FBQyxlQUFlO2dCQUN2QixHQUFHLFVBQVU7YUFDZCxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ2Q7SUFDSCxDQUFDO0lBRU8sVUFBVTtRQUNoQixFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ25ELENBQUM7Q0FDRjtBQXpHRCwwQkF5R0M7QUFFRCxTQUFTLE9BQU8sQ0FBQyxDQUFhO0lBQzVCLE9BQU8sS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7QUFDMUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHNwYXduU3luYyB9IGZyb20gJ2NoaWxkX3Byb2Nlc3MnO1xuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IGZpbmRQa2dQYXRoIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBCdWlsZGVyIG9wdGlvbnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCdWlsZGVyT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBFbnRyeSBmaWxlXG4gICAqL1xuICByZWFkb25seSBlbnRyeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgb3V0cHV0IGRpcmVjdG9yeVxuICAgKi9cbiAgcmVhZG9ubHkgb3V0RGlyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEV4cG9zZSBtb2R1bGVzIGFzIFVNRCB1bmRlciB0aGlzIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGdsb2JhbDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBNaW5pZnlcbiAgICovXG4gIHJlYWRvbmx5IG1pbmlmeT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEluY2x1ZGUgc291cmNlIG1hcHNcbiAgICovXG4gIHJlYWRvbmx5IHNvdXJjZU1hcHM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgY2FjaGUgZGlyZWN0b3J5XG4gICAqL1xuICByZWFkb25seSBjYWNoZURpcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5vZGUgdmVyc2lvbiB0byB1c2UgYXMgdGFyZ2V0IGZvciBCYWJlbFxuICAgKi9cbiAgcmVhZG9ubHkgbm9kZVZlcnNpb246IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRvY2tlciB0YWcgb2YgdGhlIG5vZGUgYmFzZSBpbWFnZSB0byB1c2UgaW4gdGhlIHBhcmNlbC1idW5kbGVyIGRvY2tlciBpbWFnZVxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vaHViLmRvY2tlci5jb20vXy9ub2RlLz90YWI9dGFnc1xuICAgKi9cbiAgcmVhZG9ubHkgbm9kZURvY2tlclRhZzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcm9vdCBvZiB0aGUgcHJvamVjdC4gVGhpcyB3aWxsIGJlIHVzZWQgYXMgdGhlIHNvdXJjZSBmb3IgdGhlIHZvbHVtZVxuICAgKiBtb3VudGVkIGluIHRoZSBEb2NrZXIgY29udGFpbmVyLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVjdFJvb3Q6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyB0byBwYXNzIHRvIHRoZSBjb250YWluZXIgcnVubmluZyBQYXJjZWwuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gZW52aXJvbm1lbnQgdmFyaWFibGVzIGFyZSBwYXNzZWQgdG8gdGhlIGNvbnRhaW5lclxuICAgKi9cbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZzsgfTtcbn1cblxuLyoqXG4gKiBCdWlsZGVyXG4gKi9cbmV4cG9ydCBjbGFzcyBCdWlsZGVyIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwa2dQYXRoOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBvcmlnaW5hbFBrZzogQnVmZmVyO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgb3JpZ2luYWxQa2dKc29uOiB7IFtrZXk6IHN0cmluZ106IGFueSB9O1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgb3B0aW9uczogQnVpbGRlck9wdGlvbnMpIHtcbiAgICAvLyBPcmlnaW5hbCBwYWNrYWdlLmpzb25cbiAgICBjb25zdCBwa2dQYXRoID0gZmluZFBrZ1BhdGgoKTtcbiAgICBpZiAoIXBrZ1BhdGgpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgYSBgcGFja2FnZS5qc29uYCBpbiB0aGlzIHByb2plY3QuJyk7XG4gICAgfVxuICAgIHRoaXMucGtnUGF0aCA9IHBhdGguam9pbihwa2dQYXRoLCAncGFja2FnZS5qc29uJyk7XG4gICAgdGhpcy5vcmlnaW5hbFBrZyA9IGZzLnJlYWRGaWxlU3luYyh0aGlzLnBrZ1BhdGgpO1xuICAgIHRoaXMub3JpZ2luYWxQa2dKc29uID0gSlNPTi5wYXJzZSh0aGlzLm9yaWdpbmFsUGtnLnRvU3RyaW5nKCkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEJ1aWxkIHdpdGggcGFyY2VsIGluIGEgRG9ja2VyIGNvbnRhaW5lclxuICAgKi9cbiAgcHVibGljIGJ1aWxkKCk6IHZvaWQge1xuICAgIHRyeSB7XG4gICAgICB0aGlzLnVwZGF0ZVBrZygpO1xuXG4gICAgICBjb25zdCBkb2NrZXJCdWlsZEFyZ3MgPSBbXG4gICAgICAgICdidWlsZCcsXG4gICAgICAgICctLWJ1aWxkLWFyZycsIGBOT0RFX1RBRz0ke3RoaXMub3B0aW9ucy5ub2RlRG9ja2VyVGFnfWAsXG4gICAgICAgICctdCcsICdwYXJjZWwtYnVuZGxlcicsXG4gICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9wYXJjZWwtYnVuZGxlcicpLFxuICAgICAgXTtcblxuICAgICAgY29uc3QgYnVpbGQgPSBzcGF3blN5bmMoJ2RvY2tlcicsIGRvY2tlckJ1aWxkQXJncyk7XG5cbiAgICAgIGlmIChidWlsZC5lcnJvcikge1xuICAgICAgICB0aHJvdyBidWlsZC5lcnJvcjtcbiAgICAgIH1cblxuICAgICAgaWYgKGJ1aWxkLnN0YXR1cyAhPT0gMCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFtTdGF0dXMgJHtidWlsZC5zdGF0dXN9XSBzdGRvdXQ6ICR7YnVpbGQuc3Rkb3V0Py50b1N0cmluZygpLnRyaW0oKX1cXG5cXG5cXG5zdGRlcnI6ICR7YnVpbGQuc3RkZXJyPy50b1N0cmluZygpLnRyaW0oKX1gKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgY29udGFpbmVyUHJvamVjdFJvb3QgPSAnL3Byb2plY3QnO1xuICAgICAgY29uc3QgY29udGFpbmVyT3V0RGlyID0gJy9vdXQnO1xuICAgICAgY29uc3QgY29udGFpbmVyQ2FjaGVEaXIgPSAnL2NhY2hlJztcbiAgICAgIGNvbnN0IGNvbnRhaW5lckVudHJ5UGF0aCA9IHBhdGguam9pbihjb250YWluZXJQcm9qZWN0Um9vdCwgcGF0aC5yZWxhdGl2ZSh0aGlzLm9wdGlvbnMucHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZSh0aGlzLm9wdGlvbnMuZW50cnkpKSk7XG5cbiAgICAgIGNvbnN0IGRvY2tlclJ1bkFyZ3MgPSBbXG4gICAgICAgICdydW4nLCAnLS1ybScsXG4gICAgICAgICctdicsIGAke3RoaXMub3B0aW9ucy5wcm9qZWN0Um9vdH06JHtjb250YWluZXJQcm9qZWN0Um9vdH1gLFxuICAgICAgICAnLXYnLCBgJHtwYXRoLnJlc29sdmUodGhpcy5vcHRpb25zLm91dERpcil9OiR7Y29udGFpbmVyT3V0RGlyfWAsXG4gICAgICAgIC4uLih0aGlzLm9wdGlvbnMuY2FjaGVEaXIgPyBbJy12JywgYCR7cGF0aC5yZXNvbHZlKHRoaXMub3B0aW9ucy5jYWNoZURpcil9OiR7Y29udGFpbmVyQ2FjaGVEaXJ9YF0gOiBbXSksXG4gICAgICAgIC4uLmZsYXR0ZW4oT2JqZWN0LmVudHJpZXModGhpcy5vcHRpb25zLmVudmlyb25tZW50IHx8IHt9KS5tYXAoKFtrLCB2XSkgPT4gWyctLWVudicsIGAke2t9PSR7dn1gXSkpLFxuICAgICAgICAnLXcnLCBwYXRoLmRpcm5hbWUoY29udGFpbmVyRW50cnlQYXRoKS5yZXBsYWNlKC9cXFxcL2csICcvJyksIC8vIEFsd2F5cyB1c2UgUE9TSVggcGF0aHMgaW4gdGhlIGNvbnRhaW5lclxuICAgICAgICAncGFyY2VsLWJ1bmRsZXInLFxuICAgICAgXTtcbiAgICAgIGNvbnN0IHBhcmNlbEFyZ3MgPSBbXG4gICAgICAgICdwYXJjZWwnLCAnYnVpbGQnLCBjb250YWluZXJFbnRyeVBhdGgucmVwbGFjZSgvXFxcXC9nLCAnLycpLCAvLyBBbHdheXMgdXNlIFBPU0lYIHBhdGhzIGluIHRoZSBjb250YWluZXJcbiAgICAgICAgJy0tb3V0LWRpcicsIGNvbnRhaW5lck91dERpcixcbiAgICAgICAgJy0tb3V0LWZpbGUnLCAnaW5kZXguanMnLFxuICAgICAgICAnLS1nbG9iYWwnLCB0aGlzLm9wdGlvbnMuZ2xvYmFsLFxuICAgICAgICAnLS10YXJnZXQnLCAnbm9kZScsXG4gICAgICAgICctLWJ1bmRsZS1ub2RlLW1vZHVsZXMnLFxuICAgICAgICAnLS1sb2ctbGV2ZWwnLCAnMicsXG4gICAgICAgICF0aGlzLm9wdGlvbnMubWluaWZ5ICYmICctLW5vLW1pbmlmeScsXG4gICAgICAgICF0aGlzLm9wdGlvbnMuc291cmNlTWFwcyAmJiAnLS1uby1zb3VyY2UtbWFwcycsXG4gICAgICAgIC4uLih0aGlzLm9wdGlvbnMuY2FjaGVEaXIgPyBbJy0tY2FjaGUtZGlyJywgY29udGFpbmVyQ2FjaGVEaXJdIDogW10pLFxuICAgICAgXS5maWx0ZXIoQm9vbGVhbikgYXMgc3RyaW5nW107XG5cbiAgICAgIGNvbnN0IHBhcmNlbCA9IHNwYXduU3luYygnZG9ja2VyJywgWy4uLmRvY2tlclJ1bkFyZ3MsIC4uLnBhcmNlbEFyZ3NdKTtcblxuICAgICAgaWYgKHBhcmNlbC5lcnJvcikge1xuICAgICAgICB0aHJvdyBwYXJjZWwuZXJyb3I7XG4gICAgICB9XG5cbiAgICAgIGlmIChwYXJjZWwuc3RhdHVzICE9PSAwKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgW1N0YXR1cyAke3BhcmNlbC5zdGF0dXN9XSBzdGRvdXQ6ICR7cGFyY2VsLnN0ZG91dD8udG9TdHJpbmcoKS50cmltKCl9XFxuXFxuXFxuc3RkZXJyOiAke3BhcmNlbC5zdGRlcnI/LnRvU3RyaW5nKCkudHJpbSgpfWApO1xuICAgICAgfVxuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBGYWlsZWQgdG8gYnVpbGQgZmlsZSBhdCAke3RoaXMub3B0aW9ucy5lbnRyeX06ICR7ZXJyfWApO1xuICAgIH0gZmluYWxseSB7IC8vIEFsd2F5cyByZXN0b3JlIHBhY2thZ2UuanNvbiB0byBvcmlnaW5hbFxuICAgICAgdGhpcy5yZXN0b3JlUGtnKCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIHBhY2thZ2UuanNvbiB0byBjb25maWd1cmUgUGFyY2VsXG4gICAqL1xuICBwcml2YXRlIHVwZGF0ZVBrZygpIHtcbiAgICBjb25zdCB1cGRhdGVEYXRhOiB7IFtrZXk6IHN0cmluZ106IGFueSB9ID0ge307XG4gICAgLy8gVXBkYXRlIGVuZ2luZXMubm9kZSAoQmFiZWwgdGFyZ2V0KVxuICAgIHVwZGF0ZURhdGEuZW5naW5lcyA9IHsgbm9kZTogYD49ICR7dGhpcy5vcHRpb25zLm5vZGVWZXJzaW9ufWAgfTtcblxuICAgIC8vIFdyaXRlIG5ldyBwYWNrYWdlLmpzb25cbiAgICBpZiAoT2JqZWN0LmtleXModXBkYXRlRGF0YSkubGVuZ3RoICE9PSAwKSB7XG4gICAgICBmcy53cml0ZUZpbGVTeW5jKHRoaXMucGtnUGF0aCwgSlNPTi5zdHJpbmdpZnkoe1xuICAgICAgICAuLi50aGlzLm9yaWdpbmFsUGtnSnNvbixcbiAgICAgICAgLi4udXBkYXRlRGF0YSxcbiAgICAgIH0sIG51bGwsIDIpKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHJlc3RvcmVQa2coKSB7XG4gICAgZnMud3JpdGVGaWxlU3luYyh0aGlzLnBrZ1BhdGgsIHRoaXMub3JpZ2luYWxQa2cpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGZsYXR0ZW4oeDogc3RyaW5nW11bXSkge1xuICByZXR1cm4gQXJyYXkucHJvdG90eXBlLmNvbmNhdChbXSwgLi4ueCk7XG59XG4iXX0=