# -*- coding: utf-8 -*-
"""
Wget command module.
"""
__author__ = 'Agnieszka Bylica, Michal Ernst'
__copyright__ = 'Copyright (C) 2018, Nokia'
__email__ = 'agnieszka.bylica@nokia.com, michal.ernst@nokia.com'

import re

from moler.cmd.unix.genericunix import GenericUnixCommand
from moler.exceptions import CommandFailure
from moler.exceptions import ParsingDone


class Wget(GenericUnixCommand):
    def __init__(self, connection, options, log_progress_bar=False, prompt=None, newline_chars=None, runner=None):
        super(Wget, self).__init__(connection=connection, prompt=prompt, newline_chars=newline_chars, runner=runner)

        self.options = options  # should contain URLs
        self.log_progress_bar = log_progress_bar
        self.current_percent = 0
        self.current_ret['RESULT'] = list()
        if self.log_progress_bar:
            self.current_ret['PROGRESS_LOG'] = list()

    def build_command_string(self):
        cmd = "wget {}".format(self.options)
        return cmd

    def on_new_line(self, line, is_full_line):
        if is_full_line:
            try:
                self._command_error(line)
                if self.log_progress_bar:
                    self._parse_line_progress_bar(line)
                self._parse_line_complete(line)
            except ParsingDone:
                pass
        super(Wget, self).on_new_line(line, is_full_line)

    _re_command_error = list()
    _re_command_error.append(re.compile(r"(?P<ERROR>Connecting\sto\s.*\sfailed:.*)", re.I))
    _re_command_error.append(re.compile(r"wget:\s(?P<ERROR>.*)", re.I))

    def _command_error(self, line):
        for _re_error in Wget._re_command_error:
            if self._regex_helper.search_compiled(_re_error, line):
                self.set_exception(CommandFailure(self, "ERROR: {}".format(self._regex_helper.group("ERROR"))))
                raise ParsingDone

    _re_progress_bar = re.compile(r"(?P<BAR>(?P<PERCENT>\d{1,3})%\s?\[\s*[=+>]*\s*\].+)", re.I)

    def _parse_line_progress_bar(self, line):
        if self._regex_helper.search_compiled(Wget._re_progress_bar, line):
            self.current_ret['PROGRESS_LOG'].append(self._regex_helper.group("BAR"))
            raise ParsingDone

    _re_file_saved = re.compile(r"(?P<SAVED>\d{4}-\d{2}-\d{2}\s\d{2}:\d{2}:\d{2}\s\(\d+.\d+\s\w{2}/s\)\s-\s.*)", re.I)

    def _parse_line_complete(self, line):
        if self._regex_helper.search_compiled(Wget._re_file_saved, line):
            self.current_ret['RESULT'].append(self._regex_helper.group("SAVED"))
            raise ParsingDone


COMMAND_OUTPUT = """moler@debian:~$ wget http://ftp.gnu.org/gnu/wget/wget-1.5.3.tar.gz
--2012-10-02 11:28:30--  http://ftp.gnu.org/gnu/wget/wget-1.5.3.tar.gz
Resolving ftp.gnu.org... 208.118.235.20, 2001:4830:134:3::b
Connecting to ftp.gnu.org|208.118.235.20|:80... connected.
HTTP request sent, awaiting response... 200 OK
Length: 446966 (436K) [application/x-gzip]
Saving to: wget-1.5.3.tar.gz
 0% [                                                                                                                                                                                                                                                                   ] 0           --.-K/s
 1% [=>                                                                                                                                                                                                                                                                 ] 3,448,832    994KB/s  eta 5m 17s
 2% [====>                                                                                                                                                                                                                                                              ] 6,152,192   1.56MB/s  eta 4m 22s
 3% [=======>                                                                                                                                                                                                                                                           ] 9,428,992   1.54MB/s  eta 3m 46s
 4% [=========>                                                                                                                                                                                                                                                         ] 12,476,416  1.47MB/s  eta 3m 36s
 5% [=============>                                                                                                                                                                                                                                                     ] 17,293,312  1.54MB/s  eta 3m 21s
 6% [==============>                                                                                                                                                                                                                                                    ] 18,309,120  1.50MB/s  eta 3m 21s
 7% [=================>                                                                                                                                                                                                                                                 ] 21,733,376  1.55MB/s  eta 3m 15s
 8% [===================>                                                                                                                                                                                                                                               ] 24,633,344  1.55MB/s  eta 3m 10s
 9% [======================>                                                                                                                                                                                                                                            ] 27,566,080  1.56MB/s  eta 3m 7s
10% [=========================>                                                                                                                                                                                                                                         ] 30,646,272  1.42MB/s  eta 3m 3s
11% [===========================>                                                                                                                                                                                                                                       ] 33,923,072  1.37MB/s  eta 3m 3s
12% [==============================>                                                                                                                                                                                                                                    ] 36,855,808  1.38MB/s  eta 3m 4s
13% [================================>                                                                                                                                                                                                                                  ] 39,854,080  1.45MB/s  eta 2m 59s
14% [===================================>                                                                                                                                                                                                                               ] 42,868,736  1.51MB/s  eta 2m 57s
15% [======================================>                                                                                                                                                                                                                            ] 46,145,536  1.56MB/s  eta 2m 53s
16% [========================================>                                                                                                                                                                                                                          ] 49,045,504  1.46MB/s  eta 2m 51s
17% [===========================================>                                                                                                                                                                                                                       ] 51,879,936  1.43MB/s  eta 2m 49s
18% [=============================================>                                                                                                                                                                                                                     ] 55,173,120  1.54MB/s  eta 2m 47s
19% [================================================>                                                                                                                                                                                                                  ] 58,122,240  1.55MB/s  eta 2m 43s
20% [===================================================>                                                                                                                                                                                                               ] 61,431,808  1.51MB/s  eta 2m 41s
21% [=====================================================>                                                                                                                                                                                                             ] 64,086,016  1.56MB/s  eta 2m 39s
22% [========================================================>                                                                                                                                                                                                          ] 67,411,968  1.59MB/s  eta 2m 36s
23% [==========================================================>                                                                                                                                                                                                        ] 70,279,168  1.56MB/s  eta 2m 33s
24% [=============================================================>                                                                                                                                                                                                     ] 73,572,352  1.54MB/s  eta 2m 32s
25% [===============================================================>                                                                                                                                                                                                   ] 76,341,248  1.53MB/s  eta 2m 29s
26% [==================================================================>                                                                                                                                                                                                ] 79,650,816  1.55MB/s  eta 2m 26s
27% [=====================================================================>                                                                                                                                                                                             ] 82,518,016  1.53MB/s  eta 2m 25s
28% [=======================================================================>                                                                                                                                                                                           ] 85,827,584  1.54MB/s  eta 2m 22s
29% [==========================================================================>                                                                                                                                                                                        ] 88,612,864  1.51MB/s  eta 2m 20s
30% [============================================================================>                                                                                                                                                                                      ] 91,594,752  1.51MB/s  eta 2m 19s
31% [===============================================================================>                                                                                                                                                                                   ] 94,593,024  1.55MB/s  eta 2m 16s
32% [=================================================================================>                                                                                                                                                                                 ] 97,722,368  1.16MB/s  eta 2m 16s
33% [====================================================================================>                                                                                                                                                                              ] 100,720,640 1.22MB/s  eta 2m 14s
34% [=======================================================================================>                                                                                                                                                                           ] 103,997,440 1.53MB/s  eta 2m 11s
35% [=========================================================================================>                                                                                                                                                                         ] 106,979,328 1.53MB/s  eta 2m 9s
36% [============================================================================================>                                                                                                                                                                      ] 109,862,912 1.52MB/s  eta 2m 8s
37% [==============================================================================================>                                                                                                                                                                    ] 112,959,488 1.51MB/s  eta 2m 6s
38% [=================================================================================================>                                                                                                                                                                 ] 115,990,528 1.46MB/s  eta 2m 3s
39% [====================================================================================================>                                                                                                                                                              ] 119,005,184 1.49MB/s  eta 2m 2s
40% [======================================================================================================>                                                                                                                                                            ] 122,085,376 1.53MB/s  eta 1m 59s
41% [=========================================================================================================>                                                                                                                                                         ] 125,394,944 1.56MB/s  eta 1m 56s
42% [===========================================================================================================>                                                                                                                                                       ] 128,196,608 1.53MB/s  eta 1m 55s
43% [==============================================================================================================>                                                                                                                                                    ] 131,522,560 1.53MB/s  eta 1m 53s
44% [=================================================================================================================>                                                                                                                                                 ] 134,356,992 1.48MB/s  eta 1m 50s
45% [===================================================================================================================>                                                                                                                                               ] 137,535,488 1.49MB/s  eta 1m 48s
46% [======================================================================================================================>                                                                                                                                            ] 140,468,224 1.53MB/s  eta 1m 47s
47% [========================================================================================================================>                                                                                                                                          ] 143,515,648 1.50MB/s  eta 1m 44s
48% [===========================================================================================================================>                                                                                                                                       ] 146,513,920 1.02MB/s  eta 1m 44s
49% [=============================================================================================================================>                                                                                                                                     ] 149,594,112 1.07MB/s  eta 1m 42s
50% [================================================================================================================================>                                                                                                                                  ] 152,838,144 1.52MB/s  eta 99s
51% [===================================================================================================================================>                                                                                                                               ] 155,869,184 1.53MB/s  eta 98s
52% [=====================================================================================================================================>                                                                                                                             ] 158,883,840 1.55MB/s  eta 95s
53% [========================================================================================================================================>                                                                                                                          ] 162,078,720 1.53MB/s  eta 93s
54% [===========================================================================================================================================>                                                                                                                       ] 165,093,376 1.54MB/s  eta 92s
55% [=============================================================================================================================================>                                                                                                                     ] 168,042,496 1.57MB/s  eta 89s
56% [================================================================================================================================================>                                                                                                                  ] 170,893,312 1.47MB/s  eta 87s
57% [==================================================================================================================================================>                                                                                                                ] 174,137,344 1.47MB/s  eta 86s
58% [=====================================================================================================================================================>                                                                                                             ] 177,037,312 1.53MB/s  eta 84s
59% [=======================================================================================================================================================>                                                                                                           ] 180,051,968 1.40MB/s  eta 82s
60% [==========================================================================================================================================================>                                                                                                        ] 183,132,160 1.56MB/s  eta 79s
61% [=============================================================================================================================================================>                                                                                                     ] 186,261,504 1.48MB/s  eta 77s
62% [===============================================================================================================================================================>                                                                                                   ] 189,423,616 1.49MB/s  eta 76s
63% [==================================================================================================================================================================>                                                                                                ] 192,520,192 1.40MB/s  eta 74s
64% [====================================================================================================================================================================>                                                                                              ] 195,534,848 1.04MB/s  eta 72s
65% [=======================================================================================================================================================================>                                                                                           ] 198,434,816 1.55MB/s  eta 70s
66% [==========================================================================================================================================================================>                                                                                        ] 201,515,008 1.54MB/s  eta 68s
67% [============================================================================================================================================================================>                                                                                      ] 204,480,512 1.30MB/s  eta 67s
68% [===============================================================================================================================================================================>                                                                                   ] 207,511,552 1.47MB/s  eta 64s
69% [=================================================================================================================================================================================>                                                                                 ] 210,804,736 1.55MB/s  eta 62s
70% [====================================================================================================================================================================================>                                                                              ] 213,819,392 1.47MB/s  eta 60s
71% [=======================================================================================================================================================================================>                                                                           ] 216,883,200 1.31MB/s  eta 58s
72% [=========================================================================================================================================================================================>                                                                         ] 219,897,856 1.51MB/s  eta 57s
73% [============================================================================================================================================================================================>                                                                      ] 223,010,816 1.48MB/s  eta 54s
74% [==============================================================================================================================================================================================>                                                                    ] 226,140,160 1.47MB/s  eta 52s
75% [=================================================================================================================================================================================================>                                                                 ] 229,203,968 1.57MB/s  eta 50s
76% [===================================================================================================================================================================================================>                                                               ] 231,923,712 1.50MB/s  eta 48s
77% [======================================================================================================================================================================================================>                                                            ] 235,200,512 1.56MB/s  eta 46s
78% [=========================================================================================================================================================================================================>                                                         ] 238,051,328 1.50MB/s  eta 45s
79% [===========================================================================================================================================================================================================>                                                       ] 241,180,672 1.19MB/s  eta 42s
80% [==============================================================================================================================================================================================================>                                                    ] 244,473,856 1.17MB/s  eta 41s
81% [=================================================================================================================================================================================================================>                                                 ] 247,537,664 1.57MB/s  eta 38s
82% [===================================================================================================================================================================================================================>                                               ] 250,372,096 1.47MB/s  eta 36s
83% [======================================================================================================================================================================================================================>                                            ] 253,353,984 1.46MB/s  eta 35s
84% [========================================================================================================================================================================================================================>                                          ] 256,368,640 1.54MB/s  eta 32s
85% [===========================================================================================================================================================================================================================>                                       ] 259,366,912 1.54MB/s  eta 30s
86% [=============================================================================================================================================================================================================================>                                     ] 262,561,792 1.53MB/s  eta 29s
87% [================================================================================================================================================================================================================================>                                  ] 265,674,752 1.66MB/s  eta 26s
88% [==================================================================================================================================================================================================================================>                                ] 268,607,488 1.55MB/s  eta 25s
89% [=====================================================================================================================================================================================================================================>                             ] 271,704,064 1.55MB/s  eta 22s
90% [========================================================================================================================================================================================================================================>                          ] 274,915,328 1.57MB/s  eta 20s
91% [==========================================================================================================================================================================================================================================>                        ] 277,684,224 1.50MB/s  eta 19s
92% [=============================================================================================================================================================================================================================================>                     ] 280,993,792 1.45MB/s  eta 16s
93% [===============================================================================================================================================================================================================================================>                   ] 283,910,144 1.56MB/s  eta 14s
94% [==================================================================================================================================================================================================================================================>                ] 286,990,336 1.40MB/s  eta 12s
95% [=====================================================================================================================================================================================================================================================>             ] 290,136,064 1.16MB/s  eta 11s
96% [=======================================================================================================================================================================================================================================================>           ] 293,003,264 1.64MB/s  eta 8s
97% [==========================================================================================================================================================================================================================================================>        ] 296,067,072 1.50MB/s  eta 6s
98% [============================================================================================================================================================================================================================================================>      ] 299,081,728 1.57MB/s  eta 5s
99% [===============================================================================================================================================================================================================================================================>   ] 302,424,064 1.60MB/s  eta 2s
100%[==================================================================================================================================================================================================================================================================>] 305,135,616 1.59MB/s   in 3m 19s
2012-10-02 11:28:38 (58.9 KB/s) - wget-1.5.3.tar.gz
moler@debian:~$"""

COMMAND_KWARGS = {
    'options': 'http://ftp.gnu.org/gnu/wget/wget-1.5.3.tar.gz',
    'log_progress_bar': False,
}

COMMAND_RESULT = {
    'RESULT': ['2012-10-02 11:28:38 (58.9 KB/s) - wget-1.5.3.tar.gz'],
}

COMMAND_OUTPUT_2 = """moler@debian:~$ wget -m http://users.student.com/lesson01/character.html
--2018-09-14 13:06:20--  http://users.student.com/lesson01/character.html
Connecting to 10.158.100.2:8080... connected.
Proxy request sent, awaiting response... 200 OK
Length: 3648 (3.6K) [text/html]
Saving to: 'users.student.com/lesson01/character.html'

0% [                                                                                                                                                                                                                                                                   ] 0           --.-K/s
10% [=========================>                                                                                                                                                                                                                                         ] 30,646,272  1.42MB/s  eta 3m 3s
20% [===================================================>                                                                                                                                                                                                               ] 61,431,808  1.51MB/s  eta 2m 41s
30% [============================================================================>                                                                                                                                                                                      ] 91,594,752  1.51MB/s  eta 2m 19s
40% [======================================================================================================>                                                                                                                                                            ] 122,085,376 1.53MB/s  eta 1m 59s
50% [================================================================================================================================>                                                                                                                                  ] 152,838,144 1.52MB/s  eta 99s
60% [==========================================================================================================================================================>                                                                                                        ] 183,132,160 1.56MB/s  eta 79s
70% [====================================================================================================================================================================================>                                                                              ] 213,819,392 1.47MB/s  eta 60s
80% [==============================================================================================================================================================================================================>                                                    ] 244,473,856 1.17MB/s  eta 41s
90% [========================================================================================================================================================================================================================================>                          ] 274,915,328 1.57MB/s  eta 20s
100%[==================================================================================================================================================================================================================================================================>] 305,135,616 1.59MB/s   in 3m 19s

2018-09-14 13:06:20 (210 MB/s) - 'users.student.com/lesson01/character.html' saved [3648/3648]

FINISHED --2018-09-14 13:06:20--
Total wall clock time: 0.2s
Downloaded: 1 files, 3.6K in 0s (210 MB/s)
moler@debian:~$"""

COMMAND_KWARGS_2 = {
    'options': '-m http://users.student.com/lesson01/character.html',
    'log_progress_bar': True
}

COMMAND_RESULT_2 = {
    'RESULT': ["""2018-09-14 13:06:20 (210 MB/s) - 'users.student.com/lesson01/character.html' saved [3648/3648]"""],
    'PROGRESS_LOG': [
        '0% [                                                                                                                                                                                                                                                                   ] 0           --.-K/s',
        '10% [=========================>                                                                                                                                                                                                                                         ] 30,646,272  1.42MB/s  eta 3m 3s',
        '20% [===================================================>                                                                                                                                                                                                               ] 61,431,808  1.51MB/s  eta 2m 41s',
        '30% [============================================================================>                                                                                                                                                                                      ] 91,594,752  1.51MB/s  eta 2m 19s',
        '40% [======================================================================================================>                                                                                                                                                            ] 122,085,376 1.53MB/s  eta 1m 59s',
        '50% [================================================================================================================================>                                                                                                                                  ] 152,838,144 1.52MB/s  eta 99s',
        '60% [==========================================================================================================================================================>                                                                                                        ] 183,132,160 1.56MB/s  eta 79s',
        '70% [====================================================================================================================================================================================>                                                                              ] 213,819,392 1.47MB/s  eta 60s',
        '80% [==============================================================================================================================================================================================================>                                                    ] 244,473,856 1.17MB/s  eta 41s',
        '90% [========================================================================================================================================================================================================================================>                          ] 274,915,328 1.57MB/s  eta 20s',
        '100%[==================================================================================================================================================================================================================================================================>] 305,135,616 1.59MB/s   in 3m 19s']
}
