"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter = exports.flatten = exports.handler = void 0;
const util_arn_parser_1 = require("@aws-sdk/util-arn-parser");
const auth_app_1 = require("@octokit/auth-app");
const auth_token_1 = require("@octokit/auth-token");
const auth_unauthenticated_1 = require("@octokit/auth-unauthenticated");
const rest_1 = require("@octokit/rest");
const aws_sdk_1 = require("aws-sdk");
const auth_1 = require("../auth");
exports.handler = async (event) => {
    console.log(`Request of type ${event.RequestType} received`);
    console.log("Payload: %j", event);
    const call = parse(event.ResourceProperties[event.RequestType]);
    if (!call) {
        return;
    }
    const octokitOptions = {};
    const { strategy, secret } = event.ResourceProperties.Auth;
    switch (strategy) {
        case auth_1.AuthenticationStrategy.AUTH_APP:
            const secretArn = util_arn_parser_1.parse(secret);
            const secretsManager = new aws_sdk_1.SecretsManager({ region: secretArn.region });
            const getSecretValueResponse = await secretsManager.getSecretValue({ SecretId: secret }).promise();
            // https://github.com/octokit/authentication-strategies.js/#github-app-or-installation-authentication
            octokitOptions.authStrategy = auth_app_1.createAppAuth;
            octokitOptions.auth = JSON.parse(getSecretValueResponse.SecretString);
            break;
        case auth_1.AuthenticationStrategy.AUTH_TOKEN:
            const parameterArn = util_arn_parser_1.parse(secret);
            const parameterName = parameterArn.resource.replace(/^parameter/, "");
            const ssm = new aws_sdk_1.SSM({ region: parameterArn.region });
            const getParameterResponse = await ssm.getParameter({ Name: parameterName, WithDecryption: true }).promise();
            // https://github.com/octokit/auth-token.js#readme
            octokitOptions.authStrategy = auth_token_1.createTokenAuth(getParameterResponse.Parameter.Value);
            break;
        case auth_1.AuthenticationStrategy.UNAUTHENTICATED:
        // https://github.com/octokit/auth-unauthenticated.js#createunauthenticatedauth-options
        default:
            octokitOptions.authStrategy = auth_unauthenticated_1.createUnauthenticatedAuth;
    }
    const octokit = new rest_1.Octokit(octokitOptions);
    try {
        // https://github.com/octokit/plugin-rest-endpoint-methods.js/#usage
        // @ts-ignore
        const response = await octokit.rest[call.endpoint][call.method](call.parameters);
        console.debug("Response: %j", response);
        const flattened = exports.flatten(response.data ?? {});
        const responsePath = call.physicalResourceId?.responsePath;
        const physicalResourceId = responsePath ? flattened[responsePath] : call.physicalResourceId?.id;
        return {
            PhysicalResourceId: physicalResourceId,
            Data: exports.filter(flattened, call.outputPaths),
        };
    }
    catch (e) {
        const err = e;
        if (!call.ignoreErrorCodesMatching || !new RegExp(call.ignoreErrorCodesMatching).test(err.message)) {
            throw e;
        }
    }
    return;
};
/**
 * Parse the GitHub API call passed in ResourceProperties OnCreate, OnUpdate and OnDelete
 *
 * @internal
 */
const parse = (stringified) => {
    return stringified ? JSON.parse(stringified) : undefined;
};
/**
 * Flattens a nested object and materializes the key paths dot separated.
 *
 * @internal
 */
exports.flatten = (object) => {
    const _flatten = (path, child) => {
        if ("object" == typeof child && null != child) {
            return Object.entries(child).reduce((result, [key, value]) => Object.assign(result, _flatten(path.concat([key]), value)), {});
        }
        return { [path.join(".")]: child };
    };
    return _flatten([], object);
};
/**
 * Filters objects keys that starts with key names provided in outputPaths.
 *
 * @internal
 */
exports.filter = (object, outputPaths) => {
    if (!outputPaths) {
        return object;
    }
    const result = {};
    for (const [key, value] of Object.entries(object)) {
        for (const outputPath of outputPaths) {
            if (key.startsWith(outputPath)) {
                result[key] = value;
            }
        }
    }
    return result;
};
//# sourceMappingURL=data:application/json;base64,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