# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['selenium_testing_library']

package_data = \
{'': ['*']}

install_requires = \
['selenium>3.0.0', 'typing_extensions>=4.0.0']

setup_kwargs = {
    'name': 'selenium-testing-library',
    'version': '2023.1',
    'description': 'A Python Selenium library inspired by the Testing Library',
    'long_description': '# Selenium Testing Library\n\n[![PyPI version](https://badge.fury.io/py/selenium-testing-library.svg)](https://badge.fury.io/py/selenium-testing-library)\n[![test](https://github.com/anze3db/selenium-testing-library/actions/workflows/main.yml/badge.svg)](https://github.com/anze3db/selenium-testing-library/actions/workflows/main.yml) [![codecov](https://codecov.io/gh/anze3db/selenium-testing-library/branch/main/graph/badge.svg?token=L1M7HO3DL7)](https://codecov.io/gh/anze3db/selenium-testing-library)\n\nSelenium Testing Library (STL) is a Python library implementing [Testing-Library](https://testing-library.com/) in Selenium.\n\n## Dependencies\n\n- Python 3.7, 3.8, 3.9, 3.10, 3.11\n- [`selenium`](https://pypi.org/project/selenium/) >= 3.0.0, [`typing-extensions`](https://pypi.org/project/typing-extensions/) >= 4.0.0\n## Installation\n\n```\npip install selenium-testing-library\n```\n\n## Quick Start\n\n```python\nfrom selenium import webdriver\nfrom selenium_testing_library import Screen\n\ndriver = webdriver.Chrome()\ndriver.open(\'https://google.com/\')\n\nscreen = Screen(driver)\nsearch_input = screen.find_by_title("Search")\nsearch.send_keys("Dogs")\nsearch_button = screen.find_by_text("Google Search")\nsearch_button.click()\nscreen.wait_for_stale(search_button)\n```\n\n## Finding elements\n\nSTL implements the [Queries API](https://testing-library.com/docs/queries/about) from the Testing Library. The Testing Library queries `get_by`, `query_by`, `find_by`, and the multiple element equivalents `get_all_by`, `query_all_by`, `find_all_by` are used in places where you would normally use Selenium\'s `find_element` and `find_elements` functions.\n\n The difference between the different queries (`get_by`, `query_by`, `find_by`) is whether the query will throw an error if the element was not found (`get_by`), return `None` (`query_by`) or block, wait and retry until the element is found (`find_by`).\n\n * `get_by` returns the element matched and throws an exception if zero or more than one element matches. This is the main function that we should be using to locate elements on a page.\n * `query_by` returns the element matched or `None` if no element match. It throws an exception if more than one element matches. Mostly used for asserting that an element is **not** present: `assert not screen.query_by_text("not on page")`.\n * `find_by` behaves like `get_by`, but uses a `WebDriverWait` to wait until the element is present in the DOM.\n * `get_all_by` returns a list of elements matched. It raises an exception if no elements match.\n * `query_all_by` returns a list of elements matched. It returns an empty list when no elements match.\n * `find_all_by` behaves like `get_all_by`, but uses a `WebDriverWait` to wait until the elements are present in the DOM.\n\n When an element is found the queries return a Selenium [`WebElement`](https://selenium-python.readthedocs.io/api.html#module-selenium.webdriver.remote.webelement) or a list containing Selenium [WebElement](https://selenium-python.readthedocs.io/api.html#module-selenium.webdriver.remote.webelement)s when using `get_all_by`, `query_all_by`, `find_all_by`.\n\nThe queries accept a tuple containing the [By class identifier](https://selenium-python.readthedocs.io/api.html#locate-elements-by) and the search query, so they can be used with XPath, Css or any other native Selenium selector:\n\n```python\nfrom selenium import webdriver\nfrom selenium.webdriver.common.by import By\nfrom selenium_testing_library import Screen\n\nscreen = Screen(webdriver.Chrome())\nscreen.get_by((By.CSS, ".my_class")) # Will throw an exception if the element is not found\nscreen.query_by((By.ID, "my_id")) # you can use regular tuples as if you were using Selenium\'s find_element()\nscreen.find_by((By.XPATH, "//div"), timeout=5, poll_frequency=0.5) # locators for searching through text also work\n```\n\n## Locator Classes\n\nFor convenience Locator classes can be used instead of the tuples:\n\n```python\nfrom selenium import webdriver\nfrom selenium_testing_library import Screen, locators\n\nscreen = Screen(webdriver.Chrome())\nscreen.get_by(locators.Css(".my_class")) # Will throw an exception if the element is not found\nscreen.query_by(locators.Id("my_id")) # you can use regular tuples as if you were using Selenium\'s find_element()\nscreen.find_by(locators.XPath("//div"), timeout=5, poll_frequency=0.5) # locators for searching through text also work\n```\n\n## Testing Library Selectors\n\nBesides all the Selenium native By selectors, the queries also support Testing Library\'s selectors:\n * [Role](https://testing-library.com/docs/queries/byrole)\n * [LabelText](https://testing-library.com/docs/queries/bylabeltext)\n * [PlaceholderText](https://testing-library.com/docs/queries/byplaceholdertext)\n * [Text](https://testing-library.com/docs/queries/bytext)\n * [DisplayValue](https://testing-library.com/docs/queries/bydisplayvalue)\n * [AltText](https://testing-library.com/docs/queries/byalttext)\n * [Title](https://testing-library.com/docs/queries/bytitle)\n * [TestId](https://testing-library.com/docs/queries/bytestid)\n\n```python\nfrom selenium import webdriver\nfrom selenium_testing_library import Screen, locators\n\nscreen = Screen(webdriver.Chrome())\nscreen.get_by(locators.Text("My Text"))\nscreen.query_by(locators.Role("button", pressed=True))\nscreen.find_by(locators.TestId("my-test"), timeout=5, poll_frequency=0.5) # locators for searching through text also work\n```\n## Helper functions\n\nFor convenience helper functions on the screen class are available to avoid instantiating locator classes all over the place:\n\n[`screen.get_by_role(role_name)`](https://testing-library.com/docs/queries/byrole) Queries for elements with the given role.\n[`screen.get_by_label_text(text)`](https://testing-library.com/docs/queries/bylabeltext) Queries for label elements that match the text string and return the corresponding input element.\n[`screen.get_by_placeholder_text(text)`](https://testing-library.com/docs/queries/byplaceholdertext) Queries elements with the matching placeholder attribute.\n[`screen.get_by_text(text)`](https://testing-library.com/docs/queries/bytext) Queries elements where the content matches the provided text.\n[`screen.get_by_display_value(value)`](https://testing-library.com/docs/queries/bydisplayvalue) Queries inputs, textareas, or selects with matching display value.\n[`screen.get_by_alt_text(text)`](https://testing-library.com/docs/queries/byalttext) Queries elements with the matching alt attribute.\n`screen.get_by_title(text)` Queries elements with the matching title attribute.\n`screen.get_by_test_id(value)` Queries elements matching the `data-testid` value.\n`screen.get_by_css(css)` Queries elements matching the specified css selector.\n`screen.get_by_xpath(xpath)` Queries elements matching the specified xpath selector.\n\nThere are also `query_by_*`, `find_by_*`, `get_all_by_*`, `query_all_by_*`, `find_all_by_*`  equivalents.\n\n**Note:** The selenium project has removed the `find_element_by_*` and `find_elements_by_*` helper functions in the [Selenium 4.3.0](https://github.com/SeleniumHQ/selenium/releases/tag/selenium-4.3.0) release, so I just want to state that the `screen` helper functions will never be deprecated or removed.\n\nExamples:\n\n```python\nfrom selenium import webdriver\nfrom selenium_testing_library import Screen\n\nscreen = Screen(webdriver.Chrome())\nscreen.query_by_role("role_name")\nscreen.get_by_label_text("label text")\nscreen.find_all_by_text("my text", timeout=5, poll_frequency=0.5)\nscreen.get_all_by_alt_text("alt text")\n```\n\n## Wait functions\n\n`wait_for(condition_function)` Waits until the condition function returns a truthy value.\n`wait_for_stale(element)` Waits until the element is removed from the DOM.\n\nExamples:\n\n```python\nfrom selenium import webdriver\nfrom selenium_testing_library import Screen, locators\n\nscreen = Screen(webdriver.Chrome())\n\n# Wait for the element to be clickable:\nelement = screen.get_by_text("Submit")\nscreen.wait_for(lambda _: element.is_enabled(), timeout=5, poll_frequency=0.5)\n# Wait for the element to be removed from the page:\nscreen.wait_for_stale(element)\n```\n\n## Querying within elements\n\n`Within(element)` Used to limit the query to the children of the provided element\n\nExample:\n\n```python\nfrom selenium import webdriver\nfrom selenium_testing_library import Screen, Within\n\nscreen = Screen(webdriver.Chrome())\nparent_element = screen.get_by_css(".container")\nWithin(parent_element).get_by_title("My title inside the container")\n```\n\n# Testing Playground URLs\n\nFor debugging using [testing-playground](https://testing-playground.com/), `screen` exposes `log_testing_playground_url()` which prints end returns a URL that can be opened in the browser.\n\n```python\n# log entire document to testing-playground\nurl = screen.log_testing_playground_url()\n# log a single element\nurl = screen.log_testing_playground_url(screen.get_by_text("test"))\n```\n\n# Contributing\n\nSetting up a local development environment\n\n```shell\ngit clone https://github.com/anze3db/selenium-testing-library.git && cd selenium-testing-library\npoetry install && poetry shell\n# Make sure `chromedriver` is in your PATH, download from https://chromedriver.chromium.org/downloads\n# run tests:\npytest --selenium-headless\n# run tests and display coverage info:\npytest --selenium-headless --cov=selenium_testing_library --cov-report html\n\n# To test on multiple Python versions make sure that py37, py38, py39 are\n# installed on your system and available through python3.7, python3.8,\n# python3.9. (Use pyenv and add the pyenv shims to your path\n# `export PATH=$(pyenv root)/shims:$PATH`). Then run tox:\ntox\n```\n\n# Releasing a new version\n\n```shell\nnpm run deploy\nbumpver update  # Wait and see if the CI is green\npoetry build && poetry publish\n```\n',
    'author': 'Anže Pečar',
    'author_email': 'anze@pecar.me',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/anze3db/selenium-testing-library',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
