# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


from __future__ import annotations

import threading
import time
from dataclasses import dataclass
from typing import Optional, Literal

Reason = Literal["deadline", "woken"]


@dataclass(frozen=True)
class Tick:
    """wait() 的返回信息。"""
    reason: Reason  # "deadline"：到点；"woken"：被 wake() 提前唤醒
    scheduled_monotonic: float  # 本次计划触发点（monotonic 秒）
    now_monotonic: float  # 实际返回时刻（monotonic 秒）
    lateness: float  # now - scheduled（秒），到点触发时可用于观测抖动


def _to_ns(seconds: float) -> int:
    ns = int(round(float(seconds) * 1_000_000_000))
    if ns <= 0:
        raise ValueError("period must be > 0 (too small after conversion)")
    return ns


def _ceil_div(a: int, b: int) -> int:
    # ceil(a/b) for integers, assuming b>0
    return (a + b - 1) // b


class AlignedTimer:
    """
    - 以首次 wait() 调用时刻作为基准 t0（“0 秒”）
    - 唤醒点严格对齐到：t0 + n * period（n 为整数）
    - 周期改变后，下一次唤醒对齐到新周期最近的未来整数倍
    - sleep 通过 Event.wait(timeout) 实现，可被 wake()/stop() 中断
    """

    def __init__(
            self,
            period: float,
            *,
            start_immediately: bool = True,
            skip_missed: bool = True,
            wake_event: Optional[threading.Event] = None,
            stop_event: Optional[threading.Event] = None,
    ) -> None:
        self._period_ns = _to_ns(period)
        self._start_immediately = bool(start_immediately)
        self._skip_missed = bool(skip_missed)

        self._wake_event = wake_event or threading.Event()
        self._stop_event = stop_event or threading.Event()

        self._lock = threading.Lock()
        self._wake_seq = 0  # wake() 调用次数（单调递增）
        self._wake_consumed = 0  # 已消费到的 seq

        self._initialized = False
        self._t0_ns: int = 0
        self._next_ns: int = 0

    # ---------- 对外控制 ----------

    def stop(self) -> None:
        """停止计时器：让 wait() 立即返回 None。"""
        self._stop_event.set()
        # 唤醒 wait()，避免其长时间阻塞
        self._wake_event.set()

    def is_stopped(self) -> bool:
        return self._stop_event.is_set()

    def wake(self) -> None:
        """打断当前睡眠（如果在睡眠），使 wait() 提前返回 reason='woken'。"""
        with self._lock:
            self._wake_seq += 1
            self._wake_event.set()

    def update_period(self, period: float) -> None:
        """
        更新周期（秒）。下一次唤醒将对齐到：从 t0 起按新周期的最近未来整数倍。
        注意：此方法不会自动 wake()；通常在你的循环体内调用即可。
        """
        new_period_ns = _to_ns(period)
        with self._lock:
            self._period_ns = new_period_ns
        # 重算 next（不需要持锁太久）
        if self._initialized:
            self._recompute_next_after_now()

    # ---------- 核心逻辑 ----------

    def wait(self) -> Optional[Tick]:
        """
        阻塞直到：
        - 到达计划触发点（reason='deadline'）
        - 被 wake() 提前唤醒（reason='woken'）
        - 被 stop() 停止（返回 None）
        """
        if self._stop_event.is_set():
            return None

        now_ns = time.monotonic_ns()

        # 初始化：第一次 wait() 作为 t0（0 秒）
        if not self._initialized:
            self._initialized = True
            self._t0_ns = now_ns
            if self._start_immediately:
                # 第一次不等待：认为“现在就是 0 秒触发点”
                scheduled = now_ns
                self._next_ns = self._t0_ns + self._period_ns
                now2 = time.monotonic_ns()
                return Tick(
                    reason="deadline",
                    scheduled_monotonic=scheduled / 1e9,
                    now_monotonic=now2 / 1e9,
                    lateness=(now2 - scheduled) / 1e9,
                )
            else:
                self._next_ns = self._t0_ns + self._period_ns

        # 每次 wait 前先确保 next 在未来（或刚好现在），并对齐到整数倍
        self._normalize_next()

        # 若已有未消费 wake（比如外部提前 set 过），直接返回 woken
        if self._consume_pending_wake_if_any():
            now2 = time.monotonic_ns()
            scheduled = self._next_ns
            return Tick(
                reason="woken",
                scheduled_monotonic=scheduled / 1e9,
                now_monotonic=now2 / 1e9,
                lateness=(now2 - scheduled) / 1e9,
            )

        # 开始等待：采用“序号 + Event”避免 clear 引发的丢信号
        with self._lock:
            seq_snapshot = self._wake_seq
            # 清掉旧的 set，保证 wait 真正阻塞；如果清掉期间有新 wake，会改变 _wake_seq
            self._wake_event.clear()

        # 计算超时
        while True:
            if self._stop_event.is_set():
                return None

            now_ns = time.monotonic_ns()
            remaining_ns = self._next_ns - now_ns
            if remaining_ns <= 0:
                # 到点（或已经略过点），按 deadline 处理并推进到下一整数倍
                scheduled = self._next_ns
                self._next_ns = scheduled + self._period_ns
                # 若 period 变化导致 next 不再对齐，下一次 wait() 会 normalize
                now2 = time.monotonic_ns()
                return Tick(
                    reason="deadline",
                    scheduled_monotonic=scheduled / 1e9,
                    now_monotonic=now2 / 1e9,
                    lateness=(now2 - scheduled) / 1e9,
                )

            # 等待到点或被唤醒
            self._wake_event.wait(remaining_ns / 1e9)

            if self._stop_event.is_set():
                return None

            # 检查是否发生新的 wake（用序号判断，避免 clear/set 竞态）
            with self._lock:
                if self._wake_seq != seq_snapshot:
                    # 消费到最新 wake
                    self._wake_consumed = self._wake_seq
                    # 如果没有更多未消费 wake，把 event 清掉，避免下次无故立即返回
                    self._wake_event.clear()

                    now2 = time.monotonic_ns()
                    scheduled = self._next_ns
                    return Tick(
                        reason="woken",
                        scheduled_monotonic=scheduled / 1e9,
                        now_monotonic=now2 / 1e9,
                        lateness=(now2 - scheduled) / 1e9,
                    )
                # 否则：可能是 timeout 或 spurious wake，继续循环检查 remaining

    # ---------- 内部辅助 ----------

    def _consume_pending_wake_if_any(self) -> bool:
        with self._lock:
            if self._wake_seq > self._wake_consumed:
                self._wake_consumed = self._wake_seq
                self._wake_event.clear()
                return True
            return False

    def _normalize_next(self) -> None:
        """
        保证 _next_ns 是从 t0 起按 period 的整数倍对齐，并且不落在过去（按 skip_missed 策略处理）。
        """
        now_ns = time.monotonic_ns()
        period_ns = self._period_ns
        t0 = self._t0_ns

        # 计算 now 对应的“最近未来整数倍”边界：t0 + ceil((now-t0)/period)*period
        if now_ns <= t0:
            k = 0
        else:
            k = _ceil_div(now_ns - t0, period_ns)

        candidate = t0 + k * period_ns

        if self._skip_missed:
            # next 直接跳到最近未来整数倍（candidate >= now）
            if candidate < now_ns:
                candidate += period_ns
            self._next_ns = candidate if self._next_ns < candidate else self._next_ns
            # 注意：这里取 max，是为了不把已经安排得更远的 next 拉回去（除非你希望“强制就近”，可改为直接赋值）
            # 但你要求“周期变化时下一次最近整数倍唤醒”，这个在 update_period() 里会重算 next。
        else:
            # 不跳过：若 next 已过，则让 next=now（随后推进 +period）
            if self._next_ns <= now_ns:
                self._next_ns = now_ns

        # 若 next 仍在过去（例如 period 被改小且 next 太旧），再兜底推进到未来整数倍
        if self._next_ns < now_ns and self._skip_missed:
            k2 = _ceil_div(now_ns - t0, period_ns)
            self._next_ns = t0 + k2 * period_ns

    def _recompute_next_after_now(self) -> None:
        """
        周期改变后：下一次唤醒强制对齐到新周期最近的未来整数倍（“就近整数倍”）。
        """
        now_ns = time.monotonic_ns()
        with self._lock:
            period_ns = self._period_ns
        t0 = self._t0_ns

        k = 0 if now_ns <= t0 else _ceil_div(now_ns - t0, period_ns)
        next_ns = t0 + k * period_ns
        if next_ns <= now_ns:
            next_ns += period_ns
        self._next_ns = next_ns
