# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


import base64
import hashlib
import hmac
import json
import threading
import time
import urllib.parse
import uuid
from datetime import datetime, timedelta
from typing import List

import dingtalk_stream
import requests
from alibabacloud_dingtalk.card_1_0 import models as dingtalkcard__1__0_models
from alibabacloud_dingtalk.card_1_0.client import Client as dingtalkcard_1_0Client
from alibabacloud_dingtalk.oauth2_1_0 import models as dingtalk_oauth2_models
from alibabacloud_dingtalk.oauth2_1_0.client import Client as DingTalkOAuth2Client
from alibabacloud_dingtalk.robot_1_0 import models as dingtalkrobot__1__0_models
from alibabacloud_dingtalk.robot_1_0.client import Client as dingtalkrobot_1_0Client
from alibabacloud_tea_openapi import models as open_api_models
from alibabacloud_tea_util import models as util_models
from dingtalk_stream import AckMessage
from django.core.cache import BaseCache
from loguru import logger
from qbtools import app_now_time
from qbtools import datacls
from qbtools import get_f_time
from qbtools.json import cvt_value_to_str
from redis import Redis

from app import CACHE
from .types import CardStatus
from .types import CheckItem, PlcCardParams, CardParamMap


class ChatBotHandler(dingtalk_stream.ChatbotHandler):
    def __init__(self, logger=logger):
        super().__init__()
        if logger:
            self.logger = logger

    async def process(self, callback: dingtalk_stream.CallbackMessage):
        incoming_message = dingtalk_stream.ChatbotMessage.from_dict(callback.data)
        content = (incoming_message.text.content or "").strip()
        self.logger.info(f"received message: {content}")
        self.logger.info(f"received message: {callback.data}")

        # 卡片模板 ID
        card_template_id = "615519f5-836a-4106-837c-7ac84a84c4f0.schema"
        # 卡片公有数据，非字符串类型的卡片数据参考文档: https://open.dingtalk.com/document/orgapp/instructions-for-filling-in-api-card-data
        card_data = {
            "markdown": content,
            "submitted": False,
            "title": "钉钉互动卡片",
            "tag": "标签",
        }

        card_instance = dingtalk_stream.CardReplier(
            self.dingtalk_client, incoming_message
        )
        # 创建并投放卡片: https://open.dingtalk.com/document/isvapp/create-and-deliver-cards
        card_instance_id = card_instance.create_and_deliver_card(
            card_template_id,
            cvt_value_to_str(card_data),
        )

        self.logger.info(f"reply card: {card_instance_id} {card_data}")

        # 更新卡片: https://open.dingtalk.com/document/isvapp/interactive-card-update-interface
        time.sleep(2)
        update_card_data = {"tag": "更新后的标签"}
        card_instance.put_card_data(
            card_instance_id,
            cvt_value_to_str(update_card_data),
            cardUpdateOptions={"updateCardDataByKey": True},
        )
        self.logger.info(f"update card: {card_instance_id} {update_card_data}")

        return AckMessage.STATUS_OK, "OK"


class CardTracker:
    def __init__(self, out_track_id) -> None:
        self.out_track_id = out_track_id

    def wait_for_confirm(self, timeout: int = 30) -> dict | bool:
        """
        等待卡片的回调
        """
        redis = CACHE.message_sync.client.get_client(write=True)
        msg = redis.brpop(self.out_track_id, timeout=timeout)  # -> (key, value) or None
        if not msg:
            return False
        data = json.loads(msg[1])
        return data


class DingtalkOldRobot:
    def __init__(self, app_secret, access_token):
        self.app_secret = app_secret
        self.access_token = access_token
        self.api_url = (
            f"https://oapi.dingtalk.com/robot/send?access_token={self.access_token}"
        )

    # 计算签名
    def get_timestamp_sign(self):
        timestamp = str(round(time.time() * 1000))
        secret_enc = self.app_secret.encode("utf-8")
        string_to_sign = "{}\n{}".format(timestamp, self.app_secret)
        string_to_sign_enc = string_to_sign.encode("utf-8")
        hmac_code = hmac.new(
            secret_enc, string_to_sign_enc, digestmod=hashlib.sha256
        ).digest()
        sign = urllib.parse.quote_plus(base64.b64encode(hmac_code))
        return timestamp, sign

    # 获取签名计算后的链接
    def get_signed_url(self):
        timestamp, sign = self.get_timestamp_sign()
        webhook = self.api_url + "&timestamp=" + timestamp + "&sign=" + sign
        return webhook

    # 定义webhook消息模式
    def get_webhook(self, mode):
        if mode == 0:  # 仅关键字
            webhook = self.api_url
        elif mode == 1 or mode == 2:  # 关键字+加签 或 关键字+加签+ip
            webhook = self.get_signed_url()
        else:
            webhook = ""
            logger.error("error! mode:   ", mode, "  webhook :  ", webhook)
        logger.info(webhook)
        return webhook

    def _get_message(self, text, user_info):
        message = {
            "msgtype": "text",  # 有text, "markdown"、link、整体跳转ActionCard 、独立跳转ActionCard、FeedCard类型等
            "text": {"content": text},  # 消息内容
            "at": {
                "atMobiles": [
                    user_info,
                ],
                "isAtAll": True,  # 是否是发送群中全体成员
            },
        }
        return message

    def send_message(self, text, user_info):
        """
        model: 0 --> 关键字; 1 --> 关键字 +加签; 2 --> 关键字+加签+IP
        :param text:
        :param user_info:
        :return:
        """
        webhook = self.get_webhook(1)
        # 构建请求头部
        header = {"Content-Type": "application/json", "Charset": "UTF-8"}
        message = self._get_message(text, user_info)
        message_json = json.dumps(message)
        info = requests.post(url=webhook, data=message_json, headers=header).json()
        code = info["errcode"]
        errmsg = info["errmsg"]
        now_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
        if code == 0:
            logger.info(
                now_time,
                ":Message sent successfully, return info:{} {}\n".format(code, errmsg),
            )
        else:
            logger.info(
                now_time
                + ":Message sending failed, return info:{} {}\n".format(code, errmsg)
            )


class DingTalkRobotBase:
    _access_token_info = None

    def __init__(self, name, app_key: str, app_secret: str):
        self.app_key = app_key
        self.app_secret = app_secret
        self.name = name

    def create_auth_client(self) -> DingTalkOAuth2Client:
        config = open_api_models.Config()
        config.protocol = "https"
        config.region_id = "central"
        return DingTalkOAuth2Client(config)

    def create_robot_client(self) -> dingtalkrobot_1_0Client:
        # 创建机器人客户端
        config = open_api_models.Config()
        config.protocol = "https"
        config.region_id = "central"
        client = dingtalkrobot_1_0Client(config)
        return client

    def create_card_client(self) -> dingtalkcard_1_0Client:
        """
        使用 Token 初始化账号Client
        @return: Client
        @throws Exception
        """
        config = open_api_models.Config()
        config.protocol = "https"
        config.region_id = "central"
        return dingtalkcard_1_0Client(config)

    def send_text_message(self, content: str, conv_id: str):
        if conv_id is None:
            raise Exception("open_id is None")

        client = self.create_robot_client()
        org_group_send_headers = dingtalkrobot__1__0_models.OrgGroupSendHeaders()
        org_group_send_headers.x_acs_dingtalk_access_token = self.access_token

        org_group_send_request = dingtalkrobot__1__0_models.OrgGroupSendRequest(
            msg_param=json.dumps({"content": content}),
            msg_key="sampleText",
            open_conversation_id=conv_id,
            robot_code=self.app_key,
        )
        try:
            client.org_group_send_with_options(
                org_group_send_request,
                org_group_send_headers,
                util_models.RuntimeOptions(),
            )
        except Exception as err:
            logger.exception(err)

    def send_card_message(
            self, out_track_id: str, card_template_id: str, card_param: CardParamMap, opensp_id: str
    ) -> CardTracker:
        if opensp_id is None:
            raise Exception("open_id is None")

        client = self.create_card_client()

        create_and_deliver_headers = dingtalkcard__1__0_models.CreateAndDeliverHeaders()
        create_and_deliver_headers.x_acs_dingtalk_access_token = self.access_token

        im_group_open_space_model = (
            dingtalkcard__1__0_models.CreateAndDeliverRequestImGroupOpenSpaceModel(
                support_forward=False,
            )
        )

        im_group_open_deliver_model = (
            dingtalkcard__1__0_models.CreateAndDeliverRequestImGroupOpenDeliverModel(
                robot_code=self.app_key
            )
        )

        card_data = dingtalkcard__1__0_models.CreateAndDeliverRequestCardData(
            card_param_map=cvt_value_to_str(datacls.to_dict(card_param, ignor_none=True))
        )

        create_and_deliver_request = dingtalkcard__1__0_models.CreateAndDeliverRequest(
            card_template_id=card_template_id,
            out_track_id=out_track_id,
            callback_type="STREAM",
            card_data=card_data,
            im_group_open_space_model=im_group_open_space_model,
            open_space_id=opensp_id,
            im_group_open_deliver_model=im_group_open_deliver_model,
        )
        logger.debug(f"create_and_deliver_request, out_track_id:{out_track_id}")

        client.create_and_deliver_with_options(
            create_and_deliver_request,
            create_and_deliver_headers,
            util_models.RuntimeOptions(),
        )
        return CardTracker(out_track_id)

    def update_card(self, out_track_id: str, card_param: CardParamMap, private_card_param: CardParamMap | None = None,
                    user_id=None):
        client = self.create_card_client()
        update_card_headers = dingtalkcard__1__0_models.UpdateCardHeaders()
        update_card_headers.x_acs_dingtalk_access_token = self.access_token
        card_update_options = (
            dingtalkcard__1__0_models.UpdateCardRequestCardUpdateOptions(
                update_card_data_by_key=True, update_private_data_by_key=False
            )
        )
        card_data = dingtalkcard__1__0_models.UpdateCardRequestCardData(
            card_param_map=cvt_value_to_str(datacls.to_dict(card_param, ignor_none=True)),
        )

        if private_card_param is None:
            private_card_param = CardParamMap()

        private_data_value_key = dingtalkcard__1__0_models.PrivateDataValue(
            card_param_map=cvt_value_to_str(datacls.to_dict(private_card_param, ignor_none=True)),
        )

        private_data = {}
        if user_id is None and len(datacls.to_dict(private_card_param, ignor_none=True)) != 0:
            raise Exception("private_card_param is not empty, but user_id is None")

        if user_id is not None:
            private_data = {
                user_id: private_data_value_key,
            }

        update_card_request = dingtalkcard__1__0_models.UpdateCardRequest(
            out_track_id=out_track_id,
            card_data=card_data,
            card_update_options=card_update_options,
            private_data=private_data,
            user_id_type=1,
        )
        return client.update_card_with_options(
            update_card_request, update_card_headers, util_models.RuntimeOptions()
        )

    @property
    def access_token(self):
        if self.__class__._access_token_info is not None:
            if datetime.now().timestamp() > self.__class__._access_token_info.get("expire_time", 0):
                self.__class__._access_token_info = None
            else:
                return self.__class__._access_token_info["access_token"]

        client = self.create_auth_client()
        get_access_token_request = dingtalk_oauth2_models.GetAccessTokenRequest(
            app_key=self.app_key, app_secret=self.app_secret
        )
        response = client.get_access_token(get_access_token_request)
        _now = datetime.now().timestamp()
        self.__class__._access_token_info = {
            "access_token": response.body.access_token,
            "expire_time": response.body.expire_in + _now,
        }
        logger.debug(f"获取acess_token: {self.__class__._access_token_info["access_token"]}")
        return self.__class__._access_token_info["access_token"]


class DingtalkQingboRobot(DingTalkRobotBase):
    def __init__(self, name, app_key, app_secret, default_opensp_id, card_template_id, cache:Redis|BaseCache):
        super().__init__(
            app_key=app_key,
            app_secret=app_secret,
            name=name,
        )
        self.cache = cache
        self.opensp_id = default_opensp_id
        self.card_template_id = card_template_id
        self.conv_id = self.opensp_id.split(".")[-1]

    def _set_card_status(self, card_tracker: CardTracker, timeout: int):
        result = card_tracker.wait_for_confirm(timeout=timeout)
        real_out_track_id = card_tracker.out_track_id.replace(".clear", "")
        if not result:
            self.update_card(out_track_id=real_out_track_id,
                             card_param=PlcCardParams(status=CardStatus.TIMEOUT))
            logger.debug(
                f"out_track_id: {real_out_track_id} timeout, auto update card status to timeout"
            )

    def send_plccard_message(
            self, opt_list: List[CheckItem], timeout: int = 30, need_user_num=2, opensp_id=None, sub_title="",
            err_desc=""
    ) -> CardTracker:
        card_template_id = self.card_template_id
        out_track_id: str = f"plcchangeinform.{uuid.uuid4()}"
        _now = app_now_time()

        card_params = PlcCardParams(tagname=f"来自{self.name}",
                                    need_user_num=need_user_num,
                                    checkboxItems=opt_list,
                                    createTime=get_f_time(_now),
                                    status=CardStatus.NULL,
                                    ddlTime=get_f_time(_now + timedelta(seconds=timeout)),
                                    sub_title=sub_title,
                                    err_desc=err_desc,
                                    processedTime="",
                                    users="",
                                    )
        _opensp_id = self.opensp_id
        if opensp_id is not None:
            _opensp_id = opensp_id
        self.send_card_message(
            out_track_id=out_track_id,
            card_template_id=card_template_id,
            card_param=card_params,
            opensp_id=_opensp_id,
        )

        # 写入缓存，用来清理过期的卡片
        CACHE.message_sync.set(out_track_id, card_params)

        # 启动一个线程，用来同步刚刚过期的卡片
        threading.Thread(
            target=self._set_card_status,
            args=(
                CardTracker(f"{out_track_id}.clear"),
                timeout,
            ),
            daemon=True,
        ).start()

        return CardTracker(out_track_id)

    def send_text_message(self, content: str, conv_id=None):
        if conv_id is None:
            conv_id = self.conv_id
        return super().send_text_message(content, conv_id=conv_id)
