# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


import asyncio
import json
import re
import time
from datetime import datetime
from datetime import timedelta

import dingtalk_stream
from dingtalk_stream import AckMessage
from django.core.cache.backends.base import BaseCache
from loguru import logger
from qbtools import app_now_time
from qbtools import datacls
from qbtools import get_f_time
from qbtools import get_p_time
from qbtools.json import cvt_value_to_str
from redis.client import Redis

from .tool import DingtalkQingboRobot
from .types import CardAction
from .types import CardStatus
from .types import CheckItem, PlcCardParams, PlcUserData, CardResponse, RespCardData


class ChatBotSampleHandler(dingtalk_stream.ChatbotHandler):
    def __init__(self, logger=logger):
        super().__init__()
        if logger:
            self.logger = logger
        self.robot_service = DingtalkQingboRobot()

    def cmd_info(self, callback: dingtalk_stream.CallbackMessage):
        incoming_message = dingtalk_stream.ChatbotMessage.from_dict(callback.data)
        conv_id = incoming_message.conversation_id
        content = f"{callback.data}"
        self.robot_service.send_text_message(content=content, conv_id=conv_id)

    def cmd_plccard(self, callback: dingtalk_stream.CallbackMessage):
        incoming_message = dingtalk_stream.ChatbotMessage.from_dict(callback.data)
        conv_id = incoming_message.conversation_id
        content = incoming_message.text.content
        _res = re.findall(r"[\d\.]+", str(content))

        opt_list = [
            CheckItem(value="kdll", text="空导流量: 23 --> 243", checked=False),
            CheckItem(value="xhf", text="循环风量: 123 --> 2323", checked=False),
            CheckItem(value="zf", text="振幅: 32 --> 12", checked=False),
        ]
        opensp_id = f"dtv1.card//IM_GROUP.{conv_id}"
        _, args = self._get_cmd_and_args(content)
        timeout, need_user_num = 20, 2

        if len(args) == 1:
            timeout, need_user_num = int(args[0]), 2
        elif len(args) == 2:
            timeout, need_user_num = int(args[0]), int(args[1])

        card_tracker = self.robot_service.send_plccard_message(opt_list=opt_list,
                                                               opensp_id=opensp_id,
                                                               timeout=timeout,
                                                               need_user_num=need_user_num)
        result = card_tracker.wait_for_confirm(timeout=timeout)
        if not result:
            logger.warning("超时了")
            return
        card_resp: CardResponse = datacls.from_dict(CardResponse, result)
        card_params = datacls.from_dict(PlcCardParams, card_resp.cardData.cardParamMap)

        status = card_params.status
        if status != CardStatus.CONFIRMED:
            logger.error("没有确认")
            return
        # 假设向plc写入成功
        time.sleep(3)
        logger.info("写入plc成功")
        self.robot_service.update_card(
            out_track_id=card_tracker.out_track_id,
            card_param=PlcCardParams(status=CardStatus.PLC_WRITE_DONE)
        )

    async def _heavy_job(self, func, cb):
        loop = asyncio.get_running_loop()
        # 如果是真正的 CPU/同步阻塞任务 ⇒ run_in_executor
        await loop.run_in_executor(None, func, cb)

    def _get_cmd_and_args(self, content):
        content = content.strip()
        content = re.sub(r"\s+", " ", content)
        _arr = content.split(" ")
        cmd = _arr[0]
        args = [] if len(_arr) == 1 else _arr[1:]
        return cmd, args

    async def process(self, callback: dingtalk_stream.CallbackMessage):
        incoming_message = dingtalk_stream.ChatbotMessage.from_dict(callback.data)
        conv_id = incoming_message.conversation_id
        content = (incoming_message.text.content or "").strip()
        reply_content = f"你好呀，{incoming_message.sender_nick}！欢迎使用钉钉机器人！, 请发送 /info 查看本群信息, /plccard_example查看卡片示例"
        content = content.strip()
        if content.startswith("/"):
            cmd, _ = self._get_cmd_and_args(content)
            attr_name = f"cmd_{cmd[1:]}"
            if hasattr(self, attr_name):
                func = getattr(self, attr_name)
                if callable(func):
                    asyncio.create_task(self._heavy_job(func, cb=callback))
                    return AckMessage.STATUS_OK, "OK"
            reply_content = f"指令未知：{content}"
        self.robot_service.send_text_message(content=reply_content, conv_id=conv_id)
        return AckMessage.STATUS_OK, "OK"


class PLCCardCallbackHandler(dingtalk_stream.CallbackHandler):
    def __init__(self, cache: Redis | BaseCache,
                 name, app_key, app_secret, default_opensp_id, card_template_id,
                 user_map: dict | None = None
                 ):
        super().__init__()
        if user_map is None:
            user_map = {}
        self.cache = cache
        self.default_robot_service = DingtalkQingboRobot(name=name,
                                                         app_key=app_key,
                                                         app_secret=app_secret,
                                                         default_opensp_id=default_opensp_id,
                                                         card_template_id=card_template_id,
                                                         cache=cache
                                                         )
        if user_map is None:
            user_map = {}
        self.user_map = {} if user_map is None else user_map

    async def process(self, callback: dingtalk_stream.CallbackMessage):
        incoming_message = dingtalk_stream.CardCallbackMessage.from_dict(callback.data)
        out_track_id = incoming_message.card_instance_id
        card_type = out_track_id.split(".")[0]
        # 如果不是PLC改变的卡片类型，直接返回
        if card_type != "plcchangeinform":
            logger.debug(f"card type not plc: {card_type}")
            return AckMessage.STATUS_OK, "OK"

        logger.info(f"card callback message: {incoming_message.to_dict()}")
        params = incoming_message.content.get("cardPrivateData", {}).get("params", {})
        action = params.get("action", "")

        checked_list = params.get("checked", [])
        checkboxitems = params.get("checkboxitems", [])
        users = params.get("users", "")
        need_user_num = params.get("need_user_num", 2)
        ddlTime = params.get("ddlTime", get_f_time(datetime.now() + timedelta(minutes=5)))
        _now = datetime.now()
        ddl_time = get_p_time(ddlTime)

        checkboxitems = datacls.from_list(CheckItem, params.get("checkboxitems", []))
        new_checkitems, user_checked = [], []
        for i, ele in enumerate(checkboxitems):
            if users == "":
                ele.checked = True
            if action == CardAction.REJECT:
                ele.checked = False
            _inchecked = ele.value in checked_list
            if _inchecked:
                user_checked += [f"{i + 1}"]

            ele.checked = _inchecked and ele.checked
            new_checkitems.append(ele)

        user_checked = ",".join(user_checked)
        if len(user_checked) == 0:
            user_checked = "无"

        # 处理结果状态
        status = CardStatus.NULL
        user_status = CardStatus.NULL
        if _now > ddl_time:
            status = CardStatus.TIMEOUT
        elif action == CardAction.REJECT:
            status = CardStatus.REJECT
            user_status = CardStatus.REJECT
            user_checked = "拒绝"
        elif action == CardAction.CONFIRM:
            if users.count("[") >= need_user_num - 1:
                status = CardStatus.CONFIRMED
            user_status = CardStatus.CONFIRMED

        # 新增users
        user_id = incoming_message.user_id
        user_map = {
            "163107443826303098": "江曼",
            "121069543124235390": "新明",
            "1807583739-1588608217": "于航",
            "262964480939770308": "国奇",
        }
        uname = user_map.get(user_id, user_id)
        add_user = f"{uname}[{user_checked}]"
        if uname not in users:
            users = f"{users},{add_user}" if users != "" else add_user

        card_data = PlcCardParams(
            checkboxItems=new_checkitems,
            users=users,
            processedTime=get_f_time(app_now_time()),
            status=status,
        )

        user_private_data = PlcUserData(user_status=user_status, user_checked=user_checked)
        # 返回数据
        response = CardResponse(
            cardData=RespCardData(cardParamMap=cvt_value_to_str(datacls.to_dict(card_data, ignor_none=True))),
            userPrivateData=RespCardData(
                cardParamMap=cvt_value_to_str(datacls.to_dict(user_private_data, ignor_none=True)))
        )

        response = datacls.to_dict(response)

        # 如果进入了确定性状态：拒绝或确认,就推送给等待中的线程
        if status != "":
            if isinstance(self.cache, BaseCache):
                client = self.cache.client.get_client(write=True)
            elif isinstance(self.cache, Redis):
                client = self.cache.client()
            else:
                raise TypeError("cache must be Redis or BaseCache")

            timeout = (ddl_time - _now).seconds
            # 推送给等待中的线程；用 LPUSH + EXPIRE 防止垃圾堆积
            client.lpush(out_track_id, json.dumps(response))
            client.expire(out_track_id, timeout)

            # 推送给等待中的清理线程
            client.lpush(f"{out_track_id}.clear", json.dumps(response))
            client.expire(f"{out_track_id}.clear", timeout)

        logger.info(f"card callback status: {status}")
        return AckMessage.STATUS_OK, response
