# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+
from itertools import combinations, product
from typing import List
import uuid
import math

def get_uid():
    return str(uuid.uuid4())

def round_half(x, method="nearest"):
    """
    将 x 四舍五入到最近的 0.5 倍。
    
    method:
        "nearest"  ——  四舍五入(≥0.25 向上，否则向下)，等距时走偶数规则  
        "floor"    ——  向下取 0.5 的整数倍  
        "ceil"     ——  向上取 0.5 的整数倍
    """
    if method == "nearest":
        return round(x * 2) / 2          # 与内置 round 的“银行家”规则保持一致
    elif method == "floor":
        return math.floor(x * 2) / 2
    elif method == "ceil":
        return math.ceil(x * 2) / 2
    else:
        raise ValueError("method 只能是 'nearest' | 'floor' | 'ceil'")



def gen_block_info_by_changes(
    start_num: float,
    min_val: float,
    max_val: float,
    delta: float,
    time_len: int,
    time_step: float = 0.5,
    min_time_span: float = 1.0,
    changes: int = 2,
) -> list:
    # ---------- 参数检查 ----------
    if time_len < 1:
        raise ValueError("`time_len` must be at least 1")
    if delta == 0:
        raise ValueError("`delta` cannot be 0")
    if not (0 <= changes <= time_len - 1):
        raise ValueError("`changes` must be in 0‥n-1")

    if time_len % time_step != 0:
        raise ValueError("`time_len` cannot be divided by `time_lidu`.")

    arr_n = int(time_len // time_step)

    # 每一个差分位置 i ∈ {1, …, n-1} 表示 arr[i] – arr[i-1]
    change_p_list = list(combinations(range(1, arr_n), changes))

    # 步进的可能取值
    step_choices = (-delta, delta)

    _step = product(step_choices, repeat=changes)
    _l = product(change_p_list, _step)

    result = []
    min_arr_p_l = min_time_span / time_step

    for c_ps, deltas in _l:
        pos_to_delta = dict(zip(c_ps, deltas))
        arr = [start_num]
        last_chang_i = 0
        res_arr = [start_num]

        for i in range(1, arr_n):
            _d = pos_to_delta.get(i, 0)
            _res = arr[-1] + _d
            if _res < min_val or _res > max_val:
                break
            if _d != 0:
                if i - last_chang_i < min_arr_p_l:
                    break
                else:
                    last_chang_i = i
            arr.append(_res)
            if _d != 0:
                res_arr.append(_res)

        # 判断剩余时间是否还满足要求
        if arr_n - last_chang_i < min_arr_p_l:
            continue
        # 判断是否提前中断导致长度不足
        if len(arr) != arr_n:
            continue
        # 计算时间长度
        block, _l = [], 0
        for x in c_ps:
            block.append((x - _l) * time_step)
            _l = x
        block.append((arr_n - x) * time_step)

        # 汇总结果
        result.append({"block": block, "res_arr": res_arr, "arr_list": arr})
    return result


def gen_block_info(start_num_list: List[float],
    min_val: float,
    max_val: float,
    delta: float,
    changes_list: List[int],
    time_len: int,
    time_step: float = 0.5,
    min_time_span: float = 1.0):
    res = []
    for start_num, changes in product(start_num_list,changes_list):
        res += gen_block_info_by_changes(start_num=start_num, min_val=min_val, max_val=max_val, delta=delta, time_len=time_len,time_step=time_step, min_time_span=min_time_span, changes=changes)
    return res


def generate_possible_blocks(
    n: int, lengths: List[int], step: float = 0.5, min_val: float | None = None
) -> List[List[float]]:
    if (n / step) % 1:
        raise ValueError(f"n={n} 不能被粒度 x={step} 整除")
    if not lengths:
        return []
    units_total = int(round(n / step))
    m = max(min_val or step, step)
    base_units = math.ceil(m / step)
    all_arrays: List[List[float]] = []
    for L in lengths:
        if L <= 0:
            raise ValueError("子数组长度必须为正整数")
        need_min = base_units * L  # 最低需要的单位数
        if units_total < need_min:  # n 太小→无法满足 min_val
            continue  # 也可改抛错
        remaining = units_total - need_min  # 剩余“星”数
        # 在 remaining+L-1 槽里插 L-1 条隔板
        for cuts in combinations(range(remaining + L - 1), L - 1):
            prev = -1
            counts: List[int] = []
            for idx in cuts + (remaining + L - 1,):
                # “+base_units” 确保每段 ≥ base_units
                counts.append(idx - prev - 1 + base_units)
                prev = idx
            all_arrays.append([c * step for c in counts])
    return all_arrays


if __name__ == "__main__":
    res = []
    res += gen_block_info_by_changes(start_num=33, min_val=28, max_val=34, delta=1, time_len=10, min_time_span=1,
                                     changes=1)
    res += gen_block_info_by_changes(start_num=33, min_val=28, max_val=34, delta=1, time_len=10, min_time_span=1,
                                     changes=2)
    res += gen_block_info_by_changes(start_num=34, min_val=28, max_val=34, delta=1, time_len=10, min_time_span=1,
                                     changes=1)
    res += gen_block_info_by_changes(start_num=34, min_val=28, max_val=34, delta=1, time_len=10, min_time_span=1,
                                     changes=2)
    print(len(res))
    for ele in res[:20]:
        print(ele)

    print("\n\n")
    res = gen_block_info(start_num_list=[33, 34], min_val=28, max_val=34, delta=1, time_len=10, min_time_span=1,
                         changes_list=[1, 2])

    for ele in res[:20]:
        print(ele)
