# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


from datetime import datetime
import logging
from typing import Callable
from zoneinfo import ZoneInfo
from loguru import logger
import sys
from pathlib import Path
import re
import stackprinter
import platform


class InterceptHandler(logging.Handler):
    """
    把 logging 的 LogRecord 转成 Loguru 的日志。
    """

    def emit(self, record: logging.LogRecord) -> None:
        # 1) 尝试找到同名的 Loguru level；找不到就用原始数字
        try:
            level = logger.level(record.levelname).name
        except ValueError:
            level = record.levelno

        # 2) 为了让日志行号、文件名指向 *真正* 的调用位置，需要追溯堆栈深度
        frame = logging.currentframe().f_back
        depth = 2
        while frame and frame.f_code.co_filename == logging.__file__:
            frame = frame.f_back
            depth += 1

        # 3) 发送到 Loguru
        logger.opt(depth=depth, exception=record.exc_info).log(
            level, record.getMessage()
        )


class ReFormater:
    def __init__(self, user_time_func: Callable | None = None) -> None:
        self.tar_len = 0
        self._shorter_cnt = 0
        self._shorter_max = 0
        self.user_time_func = user_time_func

    def _add_pad(self, content):
        """
        给 content 末尾补空格，使其长度达到 self.tar_len。
        规则：
        1. “尽可能保持最小”——遇到更短的内容时，不立即缩短，而是累计；
        2. “主动变长”——遇到更长的内容立即扩展 tar_len；
        3. 若连续 8 条内容均比当前 tar_len 短，则将 tar_len 缩短为这 8 条中最大的长度，
           保证对齐且尽量减小。
        """
        _len = len(content)

        # ---- 情况 1：需要加长 ----
        if _len > self.tar_len:
            self.tar_len = _len  # 立即扩展
            self._shorter_cnt = 0  # 重置计数器
            self._shorter_max = 0
        else:
            # ---- 情况 2：内容更短 ----
            self._shorter_cnt += 1
            # 记录本轮连续区间最大值(保持对齐时使用)
            self._shorter_max = max(self._shorter_max, _len)

            # 连续 8 条都较短 → 收缩到这 8 条中的最大值
            if self._shorter_cnt >= 8:
                self.tar_len = self._shorter_max
                self._shorter_cnt = 0
                self._shorter_max = 0

        # ---- 补空格并返回 ----
        content += " " * (self.tar_len - _len)
        return content

    def _get_common(self, record):
        cwd = f"{Path.cwd()}/"
        raw_path = f"{record['file'].path}"

        if platform.system().lower() == "windows" and ("unknown" not in raw_path):
            cwd = f"{Path.cwd().drive.lower()}{cwd[2:]}".replace("\\", "/")
            raw_path = f"{Path(raw_path).drive.lower()}{raw_path[2:]}"

        third = False
        f_path = f"{raw_path}:{record['line']}".replace("\\", "/").replace(cwd, "")

        if f_path.startswith("/"):
            third = True
            _r = re.findall(r"python[\d\.]+/(.*)", f_path)
            if len(_r) > 0:
                f_path = _r[0]
        process = f"pid:{record['process']}"
        m_name = f"{record['name']}:{record['line']}"
        return f_path, process, m_name, third

    def _gen_simu_time(self, record_time: datetime):
        u_time: datetime | None = None
        if self.user_time_func:
            u_time = self.user_time_func()
        if u_time and abs((u_time - record_time).seconds) > 10:
            time_str = u_time.strftime("%Y-%m-%d %H:%M:%S.%f")[:-3]
            return f"<yellow>{time_str}</yellow>"
        return ""

    def _get_level(self, level_name: str):
        LEVEL_ABBR_1 = {
            "TRACE": "T",
            "DEBUG": "D",
            "INFO": "I",
            "SUCCESS": "S",
            "WARNING": "W",
            "ERROR": "E",
            "CRITICAL": "C",
        }
        lvl = LEVEL_ABBR_1.get(level_name, level_name[:1])
        return lvl

    def for_console(self, record):
        f_path, process, m_name, third = self._get_common(record)
        _simu_time = self._gen_simu_time(record["time"])
        info = ""
        info += f"<green>{record['time']:YYYY-MM-DD HH:mm:ss.SSS}</green> | "
        if len(_simu_time) > 0:
            info += f"{_simu_time} | "
        info += f"<level>{self._get_level(record['level'].name)}</level> | "
        # info += f"<level>{m_name}</level> | "
        info += "<level>{message}</level>\n"

        _exception = record["exception"]
        if _exception is not None and _exception.traceback is not None:
            info += "{exception}\n"

        if "<unknown>" in info:
            info = info.replace("<unknown>", "unknown")

        return info

    def for_file(self, record):
        f_path, process, m_name, third = self._get_common(record)
        _simu_time = self._gen_simu_time(record["time"])
        info = ""
        info += f"<green>{record['time']:YYYY-MM-DD HH:mm:ss.SSS}</green> | "
        if len(_simu_time) > 0:
            info += f"{_simu_time} | "
        info += f"<level>{self._get_level(record['level'].name)}</level> | "
        info += f"<cyan>{process:<10}</cyan> | "
        info += f"<cyan>{m_name}</cyan> | "
        info += f"<cyan>{f_path}</cyan> | "
        info += "<level>{message}</level>\n"

        _exceotion = record["exception"]
        if record["exception"] is not None and _exceotion.traceback is not None:
            record["extra"]["stack"] = stackprinter.format(record["exception"])
            info += "{extra[stack]}\n"

        if "<unknown>" in info:
            info = info.replace("<unknown>", "unknown")
        return info


def dump_logging_tree(show_handlers: bool = True) -> None:
    """
    打印当前 logging 体系结构：
      • 每个 logger 的级别、propagate、handler 数
      • 可选列出 handler 详情
    """
    root = logging.getLogger()
    loggers = {
        name: obj
        for name, obj in logging.root.manager.loggerDict.items()
        if isinstance(obj, logging.Logger)
    }
    loggers["root"] = root

    for name, lg in sorted(loggers.items()):
        indent = "    " * (name.count("."))
        level = logging.getLevelName(lg.level or root.level)
        print(f"{indent}{name or 'root'} [level={level} propagate={lg.propagate}]")

        if show_handlers:
            for h in lg.handlers:
                hlevel = logging.getLevelName(h.level)
                print(
                    f"{indent}    └─ {h.__class__.__name__} "
                    f"(level={hlevel}, id={id(h)})"
                )


def dump_logging_hanlder() -> None:
    """
    打印当前 logging 体系结构：
      • 每个 logger 的级别、propagate、handler 数
      • 可选列出 handler 详情
    """
    root = logging.getLogger()
    loggers = {
        name: obj
        for name, obj in logging.root.manager.loggerDict.items()
        if isinstance(obj, logging.Logger)
    }
    loggers["root"] = root

    for name, lg in sorted(loggers.items()):
        hanlders = lg.handlers
        if not hanlders or len(hanlders) == 0:
            continue
        indent = ""
        level = logging.getLevelName(lg.level or root.level)
        print(f"{indent}{name or 'root'} [level={level} propagate={lg.propagate}]")
        for h in lg.handlers:
            hlevel = logging.getLevelName(h.level)
            print(
                f"{indent}  └─ {h.__class__.__name__} " f"(level={hlevel}, id={id(h)})"
            )


def remove_all_handlers(exclude_intercept=True) -> None:
    """
    移除所有 logger 上的非 InterceptHandler，确保只剩桥接器。
    """
    from loguru import logger as _logger

    INT_ID = id(_logger)  # 粗略区分

    for lg in [logging.getLogger()] + [
        obj
        for obj in logging.root.manager.loggerDict.values()
        if isinstance(obj, logging.Logger)
    ]:
        for h in lg.handlers[:]:
            if exclude_intercept and isinstance(h, InterceptHandler):
                continue
            lg.removeHandler(h)


# Intercept logging redirection to loguru
def redirect_logging_to_loguru(level: int | str = "WARNING") -> None:
    """
    在应用启动最早期调用，让 logging 的 root logger
    只保留一个 InterceptHandler。
    """
    # Ⓐ 移除 root logger 原有 handler，避免重复打印
    root_logger = logging.getLogger()
    for h in root_logger.handlers[:]:
        root_logger.removeHandler(h)

    # Ⓑ 添加刚才写的桥接 handler
    root_logger.addHandler(InterceptHandler())
    root_logger.setLevel(level)


def inverse(x):
    try:
        1 / x
    except ZeroDivisionError:
        logger.exception("Oups...")


def setup_loguru(f_path, level="DEBUG"):
    LOG_LEVEL = level
    LOG_PATH = f_path

    logger.remove()
    re_format_1 = ReFormater()
    re_format_2 = ReFormater()

    logger.add(
        sys.stdout,
        level=LOG_LEVEL,
        colorize=True,
        format=re_format_1.for_console,
        backtrace=True,
        diagnose=True,
    )

    # 文件输出(含轮转和保留)
    logger.add(
        LOG_PATH,
        rotation="1 day",
        retention="2 months",
        level="DEBUG",
        enqueue=True,
        encoding="utf-8",
        backtrace=True,
        diagnose=True,
        format=re_format_2.for_file,
    )

    logger.debug("Every thing is ok")
    logger.info("Every thing is ok")
    # logger.warning("Every thing is ok")
    # logger.error("Every thing is ok")
    # logger.critical("Every thing is ok")


if __name__ == "__main__":
    setup_loguru(level="DEBUG", f_path="logs/app.log")
