# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


from collections.abc import Iterable
from datetime import datetime
from inspect import isfunction

import pytz
from django.conf import settings
from django.db import models
from django.db.models.fields.files import FieldFile
from django.db.models.fields.related import ForeignKey
from django.utils import timezone


def get_model_field_info(dj_model, field_info_base=None, from_name="", addtional=True):
    if field_info_base is None:
        field_info_base = {}

    if not isinstance(dj_model, models.base.ModelBase):
        raise Exception("not django model")
    if not hasattr(dj_model, "_meta"):
        raise Exception("have no attr: _meta")

    concrete_model = dj_model._meta.concrete_model
    all_fields = concrete_model._meta.local_fields

    # 先处理本类的普通字段，用来防止被子表的相同的字段覆盖
    for field in all_fields:
        column_name = field.attname
        # field_type = field.model.__name__
        field_type = field.__class__
        f_info = {
            "from_name": from_name,
            "field_type": field_type,
            "belong_model": dj_model,
            "real_name": column_name,
        }
        key_name = (
            f"{column_name}" if from_name == "" else f"{from_name}__{column_name}"
        )
        field_info_base[key_name] = f_info

    all_attrs = {}
    if addtional:
        all_attrs = vars(dj_model)

    for name, obj in all_attrs.items():
        if not hasattr(obj, "__class__") or name.startswith("_"):
            continue
        if isfunction(obj) or isinstance(obj, property):
            # 直接定义的普通字段域
            f_info = {
                "from_name": from_name,
                "field_type": obj.__class__,
                "belong_model": dj_model,
                "real_name": name,
            }
            key_name = f"{name}" if from_name == "" else f"{from_name}__{name}"
            field_info_base[key_name] = f_info

    # 再处理子表的外间字段
    for field in all_fields:
        if not isinstance(field, ForeignKey):
            continue
        fname = field.name
        if from_name:
            fname = f"{from_name}__{fname}"
        model_obj = field.related_model().__class__
        get_model_field_info(
            model_obj, field_info_base=field_info_base, from_name=fname
        )

    return field_info_base


def get_json_model(object_s, serialize=False, dtf="%Y-%m-%d %H:%M:%S"):
    def _process_value(value):
        # 处理一些特殊类型，用来方便json.dumps序列化
        if isinstance(value, timezone.datetime):
            if dtf == 'isoformat':
                value = value.astimezone(pytz.timezone(settings.TIME_ZONE)).isoformat()
            else:                
                value = value.astimezone(pytz.timezone(settings.TIME_ZONE)).strftime(dtf)

        if isinstance(value, datetime):
            if dtf == 'isoformat':
                value = value.isoformat()
            else:
                value = value.strftime(dtf)
                
        if isinstance(value, FieldFile):
            try:
                value = value.url
            except Exception:
                value = None
        return value

    def model_inst_dumps(obj):
        result = {}
        try:
            if isinstance(obj, dict):
                if not serialize:
                    return obj
                result = {}
                for name, value in obj.items():
                    result[name] = _process_value(value=value)
                return result
            if not hasattr(obj, "__class__") or isinstance(
                    obj.__class__.__base__, models.Model
            ):
                return None
            fields_info = get_model_field_info(obj.__class__)

            for name, field_info in fields_info.items():
                real_name = field_info["real_name"]
                from_name = field_info["from_name"]
                feild_type = field_info["field_type"]
                if from_name != "":
                    fnames = from_name.split("__")
                    value = getattr(obj, fnames[0])
                    if len(fnames) > 1:
                        for fname in fnames[1:]:
                            value = getattr(value, fname)
                    value = getattr(value, real_name)
                else:
                    value = getattr(obj, real_name)
                try:
                    if hasattr(value, "__call__"):
                        value = value()
                except Exception as e:
                    raise e

                # 赋值raw value
                result[name] = value
                # 如果不需要序列化，直接返回
                if not serialize:
                    continue
                value = _process_value(value=value)
                result[name] = value

        except Exception as e:
            raise e
        return result

    if isinstance(object_s, Iterable):
        if isinstance(object_s, (str, bytes)):
            raise Exception("Not a django model")
        results = []
        for m_inst in object_s:
            results.append(model_inst_dumps(m_inst))
        return results

    return model_inst_dumps(object_s)


def get_update_map(form_data, dj_model):
    fields_info = get_model_field_info(dj_model)
    ret = {}
    for k, v in form_data.items():
        k = str(k)
        v = str(v)
        if v and k in fields_info and v != "":
            ret[k] = v
    return ret
