# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


import os
import inspect
from typing import get_type_hints
from django.apps import apps
from django.db import models

FIELD_TYPE_MAP = {
    "BooleanField": "bool",
    "FloatField": "float",
    "IntegerField": "int",
    "DateTimeField": "datetime.datetime",
    "DateTimeMiliField": "datetime.datetime",
    "AutoField": "int",
    "CharField": "str",
    "TextField": "str",
    "DateField": "datetime.date",
    "IntegerField": "int",
    "FloatField": "float",
    "BooleanField": "bool",
}


def resolve_field_type(field: models.Field) -> str:
    return FIELD_TYPE_MAP.get(field.__class__.__name__, "Any")


def generate_stub_for_model(model_class: type[models.Model]) -> str:
    lines = [f"class {model_class.__name__}(models.Model):"]
    
    # 添加 Django 字段
    for field in model_class._meta.get_fields():
        if isinstance(field, models.Field):
            field_type = resolve_field_type(field)
            lines.append(f"    {field.name}: {field_type}")

    # 添加 @property 字段
    props = {
        name: obj for name, obj in inspect.getmembers(model_class)
        if isinstance(obj, property) and name !='pk'
    }

    for name, prop in props.items():
        try:
            hints = get_type_hints(prop.fget)
            ret_type = hints.get("return", "Any")
        except Exception:
            ret_type = "Any"
        
        if isinstance(ret_type, type):
            hint_str = ret_type.__name__
        else:
            hint_str = str(ret_type)
        lines.append(f"    @property")
        lines.append(f"    def {name}(self) -> {hint_str}: ...")

    if len(lines) == 1:
        lines.append("    pass")
    return "\n".join(lines)


def generate_pyi(app_label: str, output_dir: str = "stubs") -> str:
    os.makedirs(output_dir, exist_ok=True)
    output_file = os.path.join(output_dir, f"{app_label}_models.pyi")
    lines = ["import datetime", "from django.db import models", ""]

    for model in apps.get_app_config(app_label).get_models():
        lines.append(generate_stub_for_model(model))
        lines.append("")

    with open(output_file, "w") as f:
        f.write("\n".join(lines))

    print(f"✅ Stub file written to: {output_file}")
    return output_file

if __name__ == "__main__":
    generate_pyi("cdq")
