# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


from datetime import datetime
from typing import Any, Optional

from django.db import models
from loguru import logger

from qbtools.datetime import get_zoneinfo


class ConversionMixin(models.CharField):

    def _convert(self, value):
        raise NotImplementedError("Subclasses must implement this method")

    def _to_db(self, value):
        return str(value)

    def from_db_value(self, value, expression, connection):
        """
        `value` 来自数据库：str 或 None
        只在 ORM 构造对象时调用一次；raw SQL 手动赋值不会调用。
        """
        if value in (None, ''):
            return None
        try:
            return self._convert(value)
        except (TypeError, ValueError):
            # 读取到脏数据时给个兜底，也可以选择抛错
            logger.warning(
                f"{expression.alias}.{expression.field.name} from_db_value类型转换失败，使用原始值：{value}替代，请检查并修复。")
            return value

    # —— 建议：表单 & shell 写入时也会走这里 ——
    def to_python(self, value):
        """
        admin / 表单反序列化等场景也会走。
        避免重复 parse：若已经是 PhoneNumber 直接返回。
        """
        if value in (None, ''):
            return value
        if not isinstance(value, str):
            return value

        # 字符串类型
        try:
            return self._convert(value)
        except (TypeError, ValueError):
            logger.warning(
                f"{self.cached_col.alias}.{self.cached_col.field.name} to_python类型转换失败，使用原始值：{value}替代，请检查并修复。")
            return value

    # —— 必须：把 Python → 数据库 (save) ——
    def get_prep_value(self, value: Any) -> Optional[str]:
        """
        - model.save() 之前调用。
        - 返回数据库可接受的值(str / None)。
        """

        if value in (None, ""):
            return None
        if isinstance(value, str):
            return value
        try:
            return self._to_db(value)
        except (TypeError, ValueError):
            logger.warning(
                f"{self.cached_col.alias}.{self.cached_col.field.name} get_prep_value 类型转换失败，使用原始值：{value}替代，请检查并修复。")
            return value

    def get_internal_type(self):
        return "CharField"

    def db_type(self, connection):
        return "char(%s)" % self.max_length


class FloatField(ConversionMixin):
    def _convert(self, value):
        return float(value)


class IntegerField(ConversionMixin):
    def _convert(self, value):
        return int(float(value))


class BooleanField(ConversionMixin):
    def _convert(self, value):
        if str(value).lower() in ['true', 'y', '1']:
            return True
        if str(value).lower() in ['false', 'n', '0']:
            return False
        if str(value).lower() == 'none':
            return None
        raise Exception("unknown bool value")

    def _to_db(self, value):
        if str(value).lower() in ['true', 'y', '1']:
            return "true"
        if str(value).lower() in ['false', 'n', '0']:
            return "false"
        if str(value).lower() == 'none':
            return None
        raise Exception("unknown bool value")


class DateTimeField(ConversionMixin):
    def _convert(self, value):
        dd = datetime.strptime(value, "%Y-%m-%d %H:%M:%S")
        p_time = dd.replace(tzinfo=get_zoneinfo())
        return p_time

    def _to_db(self, value):
        if isinstance(value, datetime):
            return value.strftime("%Y-%m-%d %H:%M:%S")
        return str(value)


class DateTimeMiliField(ConversionMixin):
    def _convert(self, value):
        dd = datetime.strptime(value, "%Y-%m-%d %H:%M:%S.%f")
        p_time = dd.replace(tzinfo=get_zoneinfo())
        return p_time

    def _to_db(self, value):
        if isinstance(value, datetime):
            return value.strftime("%Y-%m-%d %H:%M:%S.%f")
        return str(value)


CharField = models.CharField
