# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


import dataclasses as dclss
import json
import types
from collections.abc import Iterable
from dataclasses import is_dataclass, MISSING
from typing import Any
from typing import get_origin, get_args, Union, List, Type, TypeVar

T = TypeVar("T")


def from_dict1(data_class: Type[T], data: dict) -> T:
    """
    将嵌套的 dict 转换为嵌套的 dataclass。

    Args:
        data_class (Type[Any]): 顶层 dataclass 类型。
        data (dict): 包含 dataclass 数据的嵌套 dict。

    Returns:
        Any: 转换后的 dataclass 对象。
    """
    if not is_dataclass(data_class):
        raise ValueError("Input data_class must be a dataclass type.")

    # 构造字段数据，将嵌套字典递归转换为 dataclass
    result = {}
    for attr_name in data_class.__dataclass_fields__:
        if attr_name not in data:
            continue

        attr_type = data_class.__dataclass_fields__[attr_name].type
        attr_value = data[attr_name]

        if get_origin(attr_type) is list:
            # 如果字段类型是 list，则循环转换
            result[attr_name] = []

            # 1. 如果是str， 则可能是json格式的列表
            if isinstance(attr_value, str):
                try:
                    attr_value = json.loads(attr_value)
                except Exception:
                    pass
            # 2. 如果转换完成后可以迭代
            if isinstance(attr_value, Iterable):
                for _a in attr_value:
                    result[attr_name].append(from_dict(get_args(attr_type)[0], _a))
            else:
                result[attr_name] = attr_value

        elif is_dataclass(attr_type) and isinstance(attr_value, dict):
            # 如果字段类型是 dataclass，则递归转换
            result[attr_name] = from_dict(attr_type, attr_value)
        else:
            result[attr_name] = attr_value

    return data_class(**result)


def from_list(data_class: Type[T], data: list | str) -> List[T]:
    if isinstance(data, str):
        data = json.loads(data)

    if not isinstance(data, Iterable):
        raise Exception("data 不是合法json list字符串或者不可迭代")

    result = []
    for _val in data:
        if not isinstance(_val, dict):
            raise Exception("列表元素不是dict对象")
        result.append(from_dict(data_class=data_class, data=_val))
    return result


def update_dataclass(instance: Type[T], update_dict: dict) -> T:
    """
    递归地将 update_dict 的值更新到 dataclass 实例中。
    Args:
        instance: dataclass 实例
        update_dict: 包含更新数据的字典

    Returns:
        更新后的 dataclass 实例
    """
    if not is_dataclass(instance):
        raise ValueError("The provided instance is not a dataclass instance.")

    for key, value in update_dict.items():
        if hasattr(instance, key):
            attr = getattr(instance, key)
            # 如果属性也是 dataclass，则递归更新
            if is_dataclass(attr) and isinstance(value, dict):
                update_dataclass(attr, value)
            # 如果属性是列表，且列表中可能有 dataclass
            elif isinstance(attr, list) and isinstance(value, list):
                for i, item in enumerate(attr):
                    if (
                            i < len(value)
                            and is_dataclass(item)
                            and isinstance(value[i], dict)
                    ):
                        update_dataclass(item, value[i])
                    elif i < len(value):  # 如果不是 dataclass，直接赋值
                        attr[i] = value[i]
            # 其他情况直接赋值
            else:
                setattr(instance, key, value)
    return instance


def to_dict(dcls_instanse: Any, ignor_none=False) -> dict:
    res = dcls_instanse
    if is_dataclass(dcls_instanse.__class__):
        res = dclss.asdict(dcls_instanse)
    if not isinstance(res, dict):
        return res

    _new = {}
    for key, val in res.items():
        if isinstance(val, dict):
            _new[key] = to_dict(val, ignor_none=ignor_none)
        elif isinstance(val, Iterable) and (not isinstance(val, str)):
            _cc = []
            for ele in val:
                _cc.append(to_dict(ele, ignor_none=ignor_none))
            _new[key] = _cc
        elif val is None and ignor_none:
            continue
        else:
            _new[key] = val
    return _new


def to_list(value: List) -> List[dict]:
    result = []
    values = value
    if not isinstance(value, Iterable):
        values = [value]
    for ele in values:
        result.append(dclss.asdict(ele))
    return result


def _unwrap_optional(tp):
    """去掉 Optional / Union[..., None]，返回 (核心类型, 是否可为 None)。"""
    origin = get_origin(tp)
    # 兼容 typing.Union 以及 3.10+ 的 types.UnionType
    if origin in (Union, types.UnionType):
        args = [a for a in get_args(tp) if a is not type(None)]
        if len(args) == 1:  # Optional[T]
            return args[0], True
    return tp, False  # 不是 Optional


def from_dict(
        data_class: Type[T],
        data: dict,
        *,
        deep: bool = True,
        error_on_missing: bool = True
) -> T:
    """
    Recursively convert a *nested* ``dict`` into an instance of the given
    dataclass (and any dataclass-typed sub-fields).

    Parameters
    ----------
    data_class : Type[T]
        The **root** dataclass type you want to build, e.g. ``PlcCardParams``.
    data : dict
        A mapping whose keys match the field names of ``data_class``.
        Sub-mappings will in turn be converted to their corresponding
        (nested) dataclass types; JSON-encoded lists are auto-decoded.
    deep : bool, default ``True``
        If *True*, descend recursively into every level of nested dataclass
        or ``list[dataclass]`` values.  
        If *False*, only the top level is instantiated; nested dicts/lists are
        left “as is”.
    error_on_missing : bool, default ``True``
        Controls what to do when a **required (non-Optional) field** is missing
        from *data* **and** the dataclass field has no default value/
        ``default_factory``:  
        - ``True``  → raise :class:`ValueError`  
        - ``False`` → silently fill the field with ``None``

    Returns
    -------
    T
        An instance of *data_class* populated with values from *data*.

    Raises
    ------
    ValueError
        * If ``data_class`` is **not** a dataclass type.  
        * If ``error_on_missing`` is *True* and a required field is absent.  
        * If JSON decoding of a string-encoded list fails.

    Notes
    -----
    * **PEP-604 unions** (``X | None``) are treated as ``Optional[X]``.  
    * For list-typed fields, each element is recursively converted *iff*
      the element type is itself a dataclass and the runtime value is a dict.
    * Strings that look like JSON lists (e.g. ``'[{"a":1}]'``) are
      ``json.loads``-decoded before further processing.

    Examples
    --------
    >>> @dataclass
    ... class CheckItem:
    ...     value: str
    ...     text: str
    ...     checked: bool | None = None
    ...
    >>> @dataclass
    ... class PlcCardParams:
    ...     tagname: str | None = None
    ...     checkboxItems: list[CheckItem] | None = None
    ...
    >>> payload = {
    ...     "tagname": "来自",
    ...     "checkboxItems": [
    ...         {"value": "kdll", "text": "空导流量: 23 --> 243", "checked": False},
    ...         {"value": "zf",   "text": "振幅: 32 --> 12"}   # missing 'checked'
    ...     ]
    ... }
    >>> from_dict(PlcCardParams, payload)
    PlcCardParams(tagname='来自',
                  checkboxItems=[CheckItem(value='kdll',
                                           text='空导流量: 23 --> 243',
                                           checked=False),
                                 CheckItem(value='zf',
                                           text='振幅: 32 --> 12',
                                           checked=None)])
    """
    if not is_dataclass(data_class):
        raise ValueError("Input data_class must be a dataclass type.")

    result = {}

    for f_name, f_def in data_class.__dataclass_fields__.items():
        # ---------------- 获取字段原始类型 ----------------
        raw_type = f_def.type
        attr_type, is_optional = _unwrap_optional(raw_type)

        # ---------------- 字典里有这个字段 ----------------
        if f_name in data:
            attr_val = data[f_name]
            origin = get_origin(attr_type)

            # ---- List[...] ----
            if origin in (list, List):
                elem_type = get_args(attr_type)[0]

                # a. 字符串 → json 列表
                if isinstance(attr_val, str):
                    try:
                        attr_val = json.loads(attr_val)
                    except Exception:
                        pass

                if deep and isinstance(attr_val, Iterable) and not isinstance(attr_val, (str, bytes, bytearray)):
                    new_list = []
                    for item in attr_val:
                        if is_dataclass(elem_type) and isinstance(item, dict):
                            new_list.append(from_dict(elem_type, item, deep=deep, error_on_missing=error_on_missing))
                        else:
                            new_list.append(item)
                    result[f_name] = new_list
                else:
                    result[f_name] = attr_val

            # ---- 嵌套 dataclass ----
            elif isinstance(attr_val, dict) and deep:
                _a = get_args(attr_type)
                if is_dataclass(attr_type):
                    result[f_name] = from_dict(attr_type, attr_val, deep=deep, error_on_missing=error_on_missing)
                elif len(_a)>1 and is_dataclass(_a[1]):
                    _dd = {}
                    _, _v_type = get_args(raw_type)
                    for _kk, _vv in attr_val.items():
                        _dd[_kk] = from_dict(_v_type, _vv, deep=deep, error_on_missing=error_on_missing)
                    result[f_name] = _dd
                else:
                    result[f_name] = attr_val
            # ---- 普通字段 ----
            else:
                result[f_name] = attr_val

        # ---------------- 字典里缺失该字段 ----------------
        else:
            # → 1. dataclass 定义了默认值/工厂：什么都不用做，让 dataclass 自己填
            if f_def.default is not MISSING or f_def.default_factory is not MISSING:  # noqa
                continue
            # → 2. Optional 字段：赋 None
            elif is_optional:
                result[f_name] = None
            # → 3. 非 Optional 但想兜底：也赋 None(如需严格，可改成抛错)
            else:
                if error_on_missing:  # 严格模式 → 抛错
                    raise ValueError(f"Missing required field: {f_name}")
                result[f_name] = None  # 宽松模式 → 补 None

    return data_class(**result)
