# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+

from secrets import choice
import string
from typing import Optional

def random_string(length: int, *, alphabet: Optional[str] = None) -> str:
    """
    生成指定长度的随机字符串（加密安全）。

    参数：
        length : int
            输出字符串长度，必须为非负整数。
        alphabet : Optional[str]
            可选。可为以下预设名称或自定义字符集字符串：
              - 'alnum'（默认）：A-Za-z0-9
              - 'ascii'         ：A-Za-z
              - 'digits'        ：0-9
              - 'hex'           ：0-9a-f
              - 'HEX'           ：0-9A-F
              - 'urlsafe'       ：RFC 3986 未保留字符 A-Za-z0-9-._~
              - 'punct'         ：string.punctuation
            也可直接传入自定义字符集（例如 "ABCD1234"）。

    返回：
        str：长度为 length 的随机字符串。

    异常：
        ValueError：length < 0 或字符集为空。
        TypeError ：alphabet 类型非法。
    """
    if length < 0:
        raise ValueError("length 必须是非负整数")

    PRESETS = {
        'alnum':   string.ascii_letters + string.digits,
        'ascii':   string.ascii_letters,
        'digits':  string.digits,
        'hex':     '0123456789abcdef',
        'HEX':     '0123456789ABCDEF',
        'urlsafe': string.ascii_letters + string.digits + '-._~',
        'punct':   string.punctuation,
    }

    if alphabet is None:
        chars = PRESETS['alnum']
    elif alphabet in PRESETS:
        chars = PRESETS[alphabet]
    elif isinstance(alphabet, str):
        chars = alphabet
    else:
        raise TypeError("alphabet 应为 str 或预设名")

    if not chars:
        raise ValueError("字符集不能为空")

    return ''.join(choice(chars) for _ in range(length))


if __name__ == '__main__':
    # 16 位默认（字母+数字）
    print(random_string(16))

    # 32 位十六进制（小写）
    print(random_string(32, alphabet='hex'))

    # 12 位 URL 安全字符
    print(random_string(12, alphabet='urlsafe'))

    # 使用自定义字符集
    print(random_string(8, alphabet='ABCD1234'))

