# coding=utf-8
# +----------------------------------------------------------------------+
# | 波特智控 [ 以价值驱动应用, 以AI赋能控制, 让流程工业从稳态迈向自优化 ]          |
# +----------------------------------------------------------------------+
# | Copyright (c) 2020~2025 https://www.sdqbtech.com All rights reserved.|
# +----------------------------------------------------------------------+
# | Licensed 波特智控并不是自由软件，未经许可不得使用                           |
# +----------------------------------------------------------------------+
# | Author: 波特智控研究团队 <bodecontrol-team@sdqbtech.com>                |
# +----------------------------------------------------------------------+


from dataclasses import dataclass
from typing import Iterable, List, Optional, Sequence, Tuple
from django.db import models
from django.db.models import (
    Avg, Max, Min, Case, When, F, Q, Value, FloatField, DateTimeField, Func, Aggregate
)
from django.utils import timezone as djtz
from django.db.models.manager import Manager
from enum import Enum

# ---------- TimescaleDB函数包装 ----------
class TimeBucketGapFill(Func):
    function = 'time_bucket_gapfill'
    output_field = DateTimeField()

class Interpolate(Func):
    function = 'interpolate'
    # interpolate 仅支持数值，输出 float
    output_field = FloatField()

class Locf(Func):
    function = 'locf'
    # locf 返回“输入同型”，这里在用时根据被包裹表达式推断
    # 若用于字符串，Postgres/TimescaleDB 是支持的（anyelement）
    # 在 Django 里我们让输出字段由调用端控制，不在类上硬编码


class Last(Aggregate):
    """TimescaleDB: last(value, time)"""
    function = 'last'
    template = '%(function)s(%(expressions)s)'
    # output_field 在构造时传入（数值用 FloatField，字符串用 TextField）

class Fill(Enum):
    none = 'none'
    locf = 'locf'
    interpolate = 'interpolate'

class Agg(Enum):
    avg = 'avg'
    max = 'max'
    min = 'min'
    last = 'last'

@dataclass(frozen=True)
class FieldSpec:
    point: str                 # 传感器点位名（Coking.name）
    dtype: type = float        # 逻辑类型：float 或 str
    fill: Fill = Fill.none         # none | locf | interpolate
    agg: Agg = Agg.avg           # avg | max | min | last


class DBaseModel:
    
    objects:Manager = Manager()
    @staticmethod
    def _ensure_aware(dt):
        """将 naive datetime 转为 aware（使用当前激活时区）；aware 则原样返回。"""
        if dt is None:
            return None
        return djtz.make_aware(dt, djtz.get_current_timezone()) if djtz.is_naive(dt) else dt

    @classmethod
    def columns(cls) -> List[str]:
        return [k for k, v in cls.__dict__.items() if isinstance(v, FieldSpec)]

    @classmethod
    def _specs_from_fields(cls, fields: Optional[Sequence[str]]) -> List[Tuple[str, FieldSpec]]:
        all_items = [(k, v) for k, v in cls.__dict__.items() if isinstance(v, FieldSpec)]
        if not fields:
            return all_items
        wanted = set(fields)
        out = [(k, v) for k, v in all_items if k in wanted]
        missing = wanted - {k for k, _ in out}
        if missing:
            raise ValueError(f"未知的字段: {', '.join(sorted(missing))}")
        return out

    @classmethod
    def readlatest(
        cls,
        *,
        start,                # datetime（naive/aware皆可）
        end,                  # datetime（naive/aware皆可）
        resolution: str = '1 seconds',
        desc=True,
        fields: Optional[Sequence[str]] = None,
        extra_q: Optional[Q] = None,
    ):
        """
        返回 values 风格 QuerySet（dict），键包含：
        - 'ts'：gapfill 后的桶起始时间
        - 各列名：按列规范聚合+补全后的值
        """
        # --- 1) 解析字段与时区 ---
        spec_items = cls._specs_from_fields(fields)
        if not spec_items:
            raise ValueError("未指定任何字段。")

        name_in = []
        
        for col_name, spec in spec_items:
            name_in += [spec.point]
        
        start_aware = cls._ensure_aware(start)
        end_aware   = cls._ensure_aware(end)

        # --- 2) 基础过滤 ---
        qs = cls.objects.filter(time__gte=start_aware, time__lt=end_aware)
        if name_in:
            qs = qs.filter(name__in=name_in)
        if extra_q is not None:
            qs = qs.filter(extra_q)

        # --- 3) 时间桶（避免与模型字段 time 冲突，统一命名 'ts'）---
        qs = qs.annotate(
            ts=TimeBucketGapFill(
                Value(resolution),
                F('time'),
                Value(start_aware),
                Value(end_aware),
                output_field=DateTimeField(),
            )
        ).values('ts')

        # --- 4) 列聚合 + 填充 ---
        annotations = {}
        for col_name, spec in spec_items:
            agg = spec.agg or Agg.avg
            fill = spec.fill or  Fill.none

            # 选择 value 字段（数值）或 svalue 字段（字符串）
            if spec.dtype is float:
                val_field = F('value')
                out_field = FloatField()
            elif spec.dtype is str:
                val_field = F('svalue')
                out_field = models.TextField()
            else:
                raise ValueError(f"{col_name}: 不支持的 dtype {spec.dtype}")

            # 选择聚合器
            if agg == Agg.avg:
                if spec.dtype is not float:
                    raise ValueError(f"{col_name}: 字符串列不支持 avg 聚合")
                base_agg = Avg(val_field, filter=Q(name=spec.point), output_field=out_field)
            elif agg == Agg.max:
                if spec.dtype is not float:
                    raise ValueError(f"{col_name}: 字符串列不支持 max 聚合")
                base_agg = Max(val_field, filter=Q(name=spec.point), output_field=out_field)
            elif agg == Agg.min:
                if spec.dtype is not float:
                    raise ValueError(f"{col_name}: 字符串列不支持 min 聚合")
                base_agg = Min(val_field, filter=Q(name=spec.point), output_field=out_field)
            elif agg == Agg.last:
                # TimescaleDB: last(value, time)，通过自定义 Aggregate
                base_agg = Last(
                    val_field, F('time'),
                    filter=Q(name=spec.point),
                    output_field=out_field
                )
            else:
                raise ValueError(f"{col_name}: 不支持的聚合 {agg}")

            # 选择填充（仅在 time_bucket_gapfill 之后对“聚合结果”生效）
            if fill == Fill.interpolate:
                if spec.dtype is not float:
                    raise ValueError(f"{col_name}: 字符串列不支持 interpolate 填充")
                expr = Interpolate(base_agg, output_field=FloatField())
            elif fill == Fill.locf:
                # locf(anyelement) -> anyelement，字符串也支持
                expr = Locf(base_agg, output_field=out_field)
            elif fill == Fill.none:
                expr = base_agg
            else:
                raise ValueError(f"{col_name}: 不支持的填充 {fill}")

            annotations[col_name] = expr

        _ord = 'ts'
        if desc:
            _ord = '-ts'
        qs = qs.annotate(**annotations).order_by(_ord)
        return qs

