from __future__ import annotations

import re
from datetime import datetime, timedelta
from typing import Any, Callable, Mapping, Optional, Union
from zoneinfo import ZoneInfo

import pandas as pd

from .enums import Agg

TimeLike = Union[str, datetime, None]


def ensure_aware(dt: datetime | None, tz: Union[str, ZoneInfo]) -> datetime | None:
    """naive -> 认为其属于 tz 并加上 tzinfo；aware -> 原样返回。"""
    if dt is None:
        return None
    if isinstance(tz, str):
        tz = ZoneInfo(tz)
    if dt.tzinfo is None or dt.tzinfo.utcoffset(dt) is None:
        return dt.replace(tzinfo=tz)
    return dt


def flux_time(t: TimeLike, timezone: Union[str, ZoneInfo]) -> Optional[str]:
    """把 datetime 转为 Flux 可用 RFC3339；相对时间字符串（-1h）原样返回。"""
    if t is None:
        return None
    if isinstance(t, str):
        return t
    dt = ensure_aware(t, timezone)
    if dt is None:
        return None
    return dt.isoformat()


def pandas_freq(freq: str) -> str:
    m = re.fullmatch(r"\s*(\d+)\s*([a-zA-Z]+)\s*", str(freq))
    if not m:
        raise ValueError(f"非法 freq: {freq}")
    n = int(m.group(1))
    unit = m.group(2)

    unit_map = {
        "ms": "ms",
        "s": "s",
        "m": "min",
        "h": "h",
        "d": "d",
        "w": "w",
    }
    if unit not in unit_map:
        raise ValueError(f"不支持的 freq 单位: {unit}（支持 ms/s/m/h/d/w）")
    return f"{n}{unit_map[unit]}"


def flux_agg_fn(agg: Agg) -> str:
    return "mean" if agg in (Agg.mean, Agg.avg) else agg.value


def resolve_time_like(
    t: TimeLike, now: datetime, timezone: Union[str, ZoneInfo]
) -> TimeLike:
    """将相对时间字符串（如 -2m）转换为绝对时间，其他值原样或标准化返回。"""
    if t is None:
        return None
    if isinstance(t, str):
        m = re.fullmatch(r"\s*-(\d+)\s*([smhdw])\s*", t)
        if m:
            n = int(m.group(1))
            unit = m.group(2).lower()
            mult = {"s": 1, "m": 60, "h": 3600, "d": 86400, "w": 604800}[unit]
            return now - timedelta(seconds=n * mult)
        return t
    if isinstance(t, datetime):
        return ensure_aware(t, timezone)
    return t


def cast_series(
    series: pd.Series,
    target: Optional[Union[type, Callable[[Any], Any]]],
    *,
    tz_name: Optional[Union[str, ZoneInfo]] = None,
    src_tz: Optional[Union[str, ZoneInfo]] = None,
) -> pd.Series:
    if target is None:
        return series

    if callable(target) and not isinstance(target, type):
        return series.apply(lambda v: target(v) if pd.notna(v) else v)

    def _to_bool(v: Any) -> Any:
        if pd.isna(v):
            return pd.NA
        if isinstance(v, str):
            t = v.strip().lower()
            if t in {"true", "t", "1", "yes", "y", "on"}:
                return True
            if t in {"false", "f", "0", "no", "n", "off"}:
                return False
        try:
            return bool(float(v))
        except Exception:
            return pd.NA

    if target is bool:
        return series.apply(_to_bool).astype("boolean")
    if target is int:
        return pd.to_numeric(series, errors="coerce").astype("Int64")
    if target is float:
        return pd.to_numeric(series, errors="coerce").astype(float)
    if target is str:
        return series.astype("string")
    if target in (datetime, pd.Timestamp):
        s = pd.to_datetime(series, errors="coerce")

        def _tzobj(val: Optional[Union[str, ZoneInfo]]) -> Optional[ZoneInfo]:
            if val is None:
                return None
            return ZoneInfo(val) if isinstance(val, str) else val

        src = _tzobj(src_tz) or _tzobj(tz_name)
        dst = _tzobj(tz_name)

        if getattr(s.dt, "tz", None) is None or s.dt.tz is None:
            if src:
                s = s.dt.tz_localize(src)
        if dst:
            s = s.dt.tz_convert(dst) if getattr(s.dt, "tz", None) else s.dt.tz_localize(dst)
        return s

    return series.apply(lambda v: target(v) if pd.notna(v) else v)
