from __future__ import annotations

import json
from typing import Any, Dict, Mapping, Optional, Sequence, Type, TypeVar

import pandas as pd

TInflux = TypeVar("TInflux", bound="InfluxDB2Base")


class InfluxQuerySet(Sequence[TInflux]):
    """
    延迟实例化的记录序列：按需把 DataFrame 行转换为模型实例。
    """

    def __init__(
        self,
        df: pd.DataFrame,
        model_cls: Type[TInflux],
        allowed_fields: set[str],
        tag_value_to_field: Dict[str, str],
        query: str | None = None,
    ) -> None:
        self._df = df.reset_index(drop=True)
        self._model_cls = model_cls
        self._allowed_fields = allowed_fields
        self._tag_value_to_field = tag_value_to_field
        self.query: str | None = query
        self._cache: list[Optional[TInflux]] = [None] * len(self._df)

    def __len__(self) -> int:  # type: ignore[override]
        return len(self._df)

    def _normalize_row(self, rec: Mapping[str, Any]) -> Dict[str, Any]:
        normalized: Dict[str, Any] = {}
        for key, val in rec.items():
            if key == "time":
                normalized["time"] = val
                continue
            if key in self._allowed_fields:
                normalized[key] = val
                continue
            if key in self._tag_value_to_field:
                normalized[self._tag_value_to_field[key]] = val
        return normalized

    def _ensure(self, idx: int) -> TInflux:
        cached = self._cache[idx]
        if cached is None:
            rec = self._df.iloc[idx].to_dict()
            norm = self._normalize_row(rec)
            self._cache[idx] = self._model_cls._fast_instance(norm)  # type: ignore[attr-defined]
        return self._cache[idx]

    def _subset(self, indices: Sequence[int]) -> "InfluxQuerySet[TInflux]":
        sub_df = self._df.iloc[list(indices)].reset_index(drop=True)
        subset = self.__class__(
            sub_df,
            self._model_cls,
            self._allowed_fields,
            self._tag_value_to_field,
            query=self.query,
        )
        subset._cache = [self._cache[i] for i in indices]  # type: ignore[attr-defined]
        return subset

    def __getitem__(self, idx):  # type: ignore[override]
        if isinstance(idx, slice):
            indices = range(*idx.indices(len(self)))
            return self._subset(indices)
        if idx < 0:
            idx += len(self)
        if idx < 0 or idx >= len(self):
            raise IndexError("list index out of range")
        return self._ensure(int(idx))

    def __iter__(self):  # type: ignore[override]
        for i in range(len(self)):
            yield self._ensure(i)

    def to_json(self, *, as_str: bool = True) -> str | list[Dict[str, Any]]:
        """
        序列化查询结果；as_str=True 返回 JSON 字符串，False 返回字典列表。
        """
        payload = [obj.to_json(as_str=False) for obj in self]
        if as_str:
            return json.dumps(payload, ensure_ascii=False, default=str)
        return payload

    def to_dataframe(self, *, copy: bool = True) -> pd.DataFrame:
        """
        返回底层 DataFrame；默认返回副本以避免外部修改影响缓存，可传 copy=False 直接返回内部引用。
        """
        return self._df.copy(deep=True) if copy else self._df


if False:  # pragma: no cover
    from .base import InfluxDB2Base  # noqa: F401  # for type checking
