from dataclasses import dataclass
from datetime import datetime
from typing import Any, Callable, Mapping, Optional, Union
from zoneinfo import ZoneInfo
import pandas as pd

from .enums import Agg, Fill


@dataclass(frozen=True)
class FieldSpec:
    tags: Mapping[str, str]  # tag 键值（如 {"name": "..."}）
    field: str  # 存储层 _field 名（如 value/svalue）
    dtype: Optional[Union[type, Callable[[Any], Any]]] = None  # influxdb 实际类型；默认沿用 target_type
    fill: Fill = Fill.none
    agg: Agg = Agg.mean
    label: Optional[str] = None  # 可选的展示名称/中文名
    post: Optional[Callable[[Mapping[str, Any]], Any]] = None  # 读取后进一步处理的函数（整行）
    src_tz: Optional[Union[str, ZoneInfo]] = None  # datetime 类型解析时的源时区（缺省用类时区）

    def target_type(self) -> Optional[Union[type, Callable[[Any], Any]]]:
        """读取后转换的目标类型，默认不转换。子类可覆盖。"""
        return None

    def resolved_dtype(self) -> Optional[Union[type, Callable[[Any], Any]]]:
        """返回 influxdb 实际类型，未显式指定时沿用 target_type。"""
        if self.dtype is not None:
            return self.dtype
        target = self.target_type()
        return target if isinstance(target, type) else None


@dataclass(frozen=True)
class FloatField(FieldSpec):
    agg: Agg = Agg.mean

    def target_type(self) -> type:
        return float


@dataclass(frozen=True)
class IntField(FieldSpec):
    agg: Agg = Agg.mean

    def target_type(self) -> type:
        return int


@dataclass(frozen=True)
class BoolField(FieldSpec):
    agg: Agg = Agg.last

    def target_type(self) -> type:
        return bool


@dataclass(frozen=True)
class StringField(FieldSpec):
    agg: Agg = Agg.last

    def target_type(self) -> type:
        return str


@dataclass(frozen=True)
class DateTimeField(FieldSpec):
    agg: Agg = Agg.last

    def target_type(self) -> type:
        return datetime
