from __future__ import annotations

from datetime import datetime
from typing import Any, Callable, Dict, Mapping, Optional, Sequence, Tuple, Type, TypeVar, Union
from typing import TYPE_CHECKING
from zoneinfo import ZoneInfo

import pandas as pd
from typing_extensions import dataclass_transform

from .enums import Agg, Fill
from .fields import BoolField, DateTimeField, FieldSpec, FloatField, IntField, StringField
from .queryset import InfluxQuerySet

__all__ = [
    "InfluxDB2Base"
]

if TYPE_CHECKING:  # pragma: no cover
    from influxdb_client import InfluxDBClient

TimeLike = Union[str, datetime, None]

TInflux = TypeVar("TInflux", bound="InfluxDB2Base")

@dataclass_transform(
    kw_only_default=True,
    field_specifiers=(FieldSpec, pd.Timestamp),
)
class InfluxDB2Base:
    db: str
    measurement: str
    timezone: Union[str, ZoneInfo]
    bucket: str
    readonly: bool
    time: pd.Timestamp
    _sources: Dict[str, Dict[str, str]]
    _clients: Dict[str, "InfluxDBClient"]
    _client_factory: Optional[Callable[[Dict[str, str]], "InfluxDBClient"]]
    _values: Dict[str, Any]

    def __getitem__(self, key: str) -> Any: ...

    def get(self, key: str, default: Any = ...) -> Any: ...

    def to_json(self, *, as_str: bool = ...) -> str | Dict[str, Any]: ...

    @classmethod
    def field_spec(cls, name: str) -> FieldSpec: ...

    @classmethod
    def from_row(
        cls,
        row: Mapping[str, Any] | pd.Series,
        *,
        strict: bool = True,
    ) -> "InfluxDB2Base": ...

    @classmethod
    def configure_sources(
        cls,
        sources: Mapping[str, Mapping[str, str] | Any],
        *,
        client_factory: Optional[Callable[[Dict[str, str]], "InfluxDBClient"]] = ...,
        reset_cached_clients: bool = True,
    ) -> None: ...

    @staticmethod
    def _normalize_source_conf(conf: Mapping[str, str] | Any) -> Dict[str, str]: ...

    @classmethod
    def _source_name(cls, source: Optional[str]) -> str: ...

    @classmethod
    def _get_source_conf(
        cls, source: Optional[str] = ...
    ) -> Tuple[str, Dict[str, str]]: ...

    @staticmethod
    def _default_client_factory(conf: Dict[str, str]) -> "InfluxDBClient": ...

    @classmethod
    def _get_client(
        cls, source: Optional[str] = ..., conf: Optional[Dict[str, str]] = ...
    ) -> "InfluxDBClient": ...

    @classmethod
    def _resolve_connection(
        cls,
        *,
        query_api: Any = ...,
        bucket: Optional[str] = ...,
        org: Optional[str] = ...,
        source: Optional[str] = ...,
    ) -> Tuple[Any, str, str]: ...

    @classmethod
    def _resolve_write(
        cls,
        *,
        write_api: Any = ...,
        bucket: Optional[str] = ...,
        org: Optional[str] = ...,
        source: Optional[str] = ...,
    ) -> Tuple[Any, str, str]: ...

    @classmethod
    def _ensure_aware(
        cls, dt: datetime | None, tz: Union[str, ZoneInfo]
    ) -> datetime | None: ...

    @classmethod
    def _flux_time(cls, t: TimeLike) -> Optional[str]: ...

    @staticmethod
    def _pandas_freq(freq: str) -> str: ...

    @staticmethod
    def _flux_agg_fn(agg: Agg) -> str: ...

    @staticmethod
    def _is_numeric_field(spec: FieldSpec) -> bool: ...

    @staticmethod
    def _require_tags(spec: FieldSpec) -> Mapping[str, str]: ...

    @classmethod
    def _pivot_tag_key(cls, spec_items: Sequence[tuple[str, FieldSpec]]) -> str: ...

    @staticmethod
    def _column_label(spec: FieldSpec, pivot_key: str, fallback: str) -> str: ...

    @classmethod
    def _now(cls, tz_name: Optional[str] = ...) -> datetime: ...

    @staticmethod
    def _bool_from_scalar(value: Any) -> bool: ...

    @classmethod
    def _convert_value_for_write(
        cls, value: Any, spec: FieldSpec, *, field_name: str
    ) -> Any: ...

    @classmethod
    def _normalize_time_for_write(
        cls, t: Any, tz: Optional[Union[str, ZoneInfo]]
    ) -> Any: ...

    @classmethod
    def _build_points(
        cls,
        rows: Sequence[Mapping[str, Any]] | Mapping[str, Any],
        *,
        fields: Optional[Sequence[str]] = ...,
        time_key: str = "_time",
        tz_name: Optional[str] = ...,
        default_time: Any = ...,
        strict_keys: bool = True,
    ) -> list[Any]: ...

    def _resolve_write_time(self, *, time: Any = ..., tz_name: Optional[str] = ...) -> Any: ...

    def _points_from_values(
        self,
        *,
        tz_name: Optional[str] = ...,
        time: Any = ...,
        fields: Optional[Sequence[str]] = ...,
        strict_keys: bool = True,
    ) -> Tuple[list[Any], Any]: ...

    def to_line_protocol(
        self,
        *,
        tz_name: Optional[str] = ...,
        time: Any = ...,
        fields: Optional[Sequence[str]] = ...,
        precision: str = "s",
    ) -> str: ...

    def save(
        self,
        *,
        precision: str = "s",
        tz_name: Optional[str] = ...,
        source: Optional[str] = ...,
        write_api: Any = ...,
        bucket: Optional[str] = ...,
        org: Optional[str] = ...,
        time: Any = ...,
        fields: Optional[Sequence[str]] = ...,
    ) -> str: ...

    @classmethod
    def columns(cls) -> list[str]: ...

    @classmethod
    def _specs_from_fields(
        cls, fields: Optional[Sequence[str]]
    ) -> list[tuple[str, FieldSpec]]: ...

    @classmethod
    def _property_fields(cls) -> list[str]: ...

    @classmethod
    def _normalize_record(
        cls,
        rec: Mapping[str, Any],
        allowed_fields: set[str],
        tag_value_to_field: Dict[str, str],
    ) -> Dict[str, Any]: ...

    @classmethod
    def _fast_instance(
        cls: Type[TInflux], normalized: Mapping[str, Any]
    ) -> TInflux: ...

    @classmethod
    def _get_query_str(
        cls,
        *,
        bucket: str,
        start_time: TimeLike,
        end_time: TimeLike,
        freq: str = "1s",
        fields: Optional[Sequence[str]] = ...,
        create_empty_in_flux: bool = False,
    ) -> str: ...

    @classmethod
    def _concat_dfs(cls, dfs: pd.DataFrame | list[pd.DataFrame]) -> pd.DataFrame: ...

    @classmethod
    def _gapfill(
        cls,
        df: pd.DataFrame,
        *,
        start_time: TimeLike,
        end_time: TimeLike,
        freq: str,
        fields: Optional[Sequence[str]] = ...,
        tz_name: Optional[str] = ...,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = False,
    ) -> pd.DataFrame: ...

    @classmethod
    def _read_last_df(
        cls: Type[TInflux],
        *,
        start_time: TimeLike,
        end_time: TimeLike = ...,
        freq: str = "5s",
        fields: Optional[Sequence[str]] = ...,
        create_empty_in_flux: bool = True,
        tz_name: Optional[str] = ...,
        do_gapfill: bool = True,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = True,
        now: Optional[datetime] = ...,
        tolerance: Optional[float | int] = ...,
        include_properties: bool = False,
        query_api: Any = ...,
        org: Optional[str] = ...,
        bucket: Optional[str] = ...,
        source: Optional[str] = ...,
    ) -> pd.DataFrame: ...

    @classmethod
    def read_last(
        cls: Type[TInflux],
        *,
        start_time: TimeLike,
        end_time: TimeLike = ...,
        freq: str = "5s",
        fields: Optional[Sequence[str]] = ...,
        create_empty_in_flux: bool = True,
        tz_name: Optional[str] = ...,
        do_gapfill: bool = True,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = True,
        now: Optional[datetime] = ...,
        tolerance: Optional[float | int] = ...,
        query_api: Any = ...,
        org: Optional[str] = ...,
        bucket: Optional[str] = ...,
        source: Optional[str] = ...,
        lazy: bool = True,
    ) -> InfluxQuerySet[TInflux]: ...

    @classmethod
    def read_last_dataframe(
        cls,
        *,
        start_time: TimeLike,
        end_time: TimeLike = ...,
        freq: str = "5s",
        fields: Optional[Sequence[str]] = ...,
        create_empty_in_flux: bool = True,
        tz_name: Optional[str] = ...,
        do_gapfill: bool = True,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = True,
        now: Optional[datetime] = ...,
        tolerance: Optional[float | int] = ...,
        query_api: Any = ...,
        org: Optional[str] = ...,
        bucket: Optional[str] = ...,
        source: Optional[str] = ...,
    ) -> pd.DataFrame: ...

    @classmethod
    def _resolve_time_like(cls, t: TimeLike, now: datetime) -> TimeLike: ...

    @classmethod
    def _cast_types(
        cls,
        df: pd.DataFrame,
        *,
        fields: Optional[Sequence[str]],
        rename_columns_to_fieldname: bool,
        tz_name: Optional[str] = ...,
    ) -> pd.DataFrame: ...
