import json
import numbers
from dataclasses import asdict, is_dataclass
from datetime import datetime
from typing import (
    Any,
    Callable,
    Dict,
    List,
    Mapping,
    Optional,
    Sequence,
    Tuple,
    Type,
    TypeVar,
    Union,
)
from typing import TYPE_CHECKING
from zoneinfo import ZoneInfo

import pandas as pd

try:
    from typing import dataclass_transform
except ImportError:
    from typing_extensions import dataclass_transform

if TYPE_CHECKING:  # pragma: no cover
    from influxdb_client import InfluxDBClient

from .enums import Agg, Fill
from .fields import FieldSpec, DateTimeField
from .queryset import InfluxQuerySet
from .utils import (
    cast_series,
    ensure_aware,
    flux_agg_fn,
    flux_time,
    pandas_freq,
    resolve_time_like,
)

TInflux = TypeVar("TInflux", bound="InfluxDB2Base")

TimeLike = Union[str, datetime, None]


@dataclass_transform(kw_only_default=True, field_specifiers=(FieldSpec, pd.Timestamp))
class InfluxDB2Base:
    db = "default"
    measurement = "default"
    timezone = "Asia/Shanghai"
    bucket = "defualt"
    readonly = False
    time: pd.Timestamp
    _sources: Dict[str, Dict[str, str]] = {}
    _clients: Dict[str, "InfluxDBClient"] = {}
    _client_factory: Optional[Callable[[Dict[str, str]], "InfluxDBClient"]] = None
    _values: Dict[str, Any]

    # ===== 实例化与字段访问 =====
    def __init__(self, **values: Any) -> None:
        self._values = {}
        allowed = set(self.columns()) | {"time"}
        for k, v in values.items():
            if k not in allowed:
                continue
            self._values[k] = v

    def __getattribute__(self, name: str) -> Any:
        # 优先返回实例字段值，避免落到类属性（FieldSpec）上
        if not name.startswith("_"):
            try:
                values = object.__getattribute__(self, "_values")
            except AttributeError:
                values = None
            if isinstance(values, dict) and name in values:
                return values[name]
        return object.__getattribute__(self, name)

    def __getitem__(self, key: str) -> Any:
        return self._values[key]

    def get(self, key: str, default: Any = None) -> Any:
        return self._values.get(key, default)

    def to_json(self, *, as_str: bool = True) -> str | Dict[str, Any]:
        """
        序列化当前实例；as_str=True 返回 JSON 字符串，False 返回字典。
        """
        data: Dict[str, Any] = dict(self._values)
        cls = self.__class__
        for prop in cls._property_fields():
            if prop not in data:
                data[prop] = getattr(self, prop)
        if as_str:
            return json.dumps(data, ensure_ascii=False, default=str)
        return data

    @classmethod
    def field_spec(cls, name: str) -> FieldSpec:
        items = {k: v for k, v in cls.__dict__.items() if isinstance(v, FieldSpec)}
        if name not in items:
            raise ValueError(f"未知字段: {name}")
        return items[name]

    @classmethod
    def from_row(
        cls, row: Union[Mapping[str, Any], pd.Series], *, strict: bool = True
    ) -> "InfluxDB2Base":
        """
        根据字典/Series 创建实例，字段名应使用类属性名。strict=True 时未知字段会报错。
        """
        if isinstance(row, pd.Series):
            data = row.to_dict()
        elif isinstance(row, Mapping):
            data = dict(row)
        else:
            raise TypeError("row 需为 Mapping 或 pandas.Series")

        if strict:
            allowed = set(cls.columns()) | {"_time"}
            unknown = set(data) - allowed
            if unknown:
                raise ValueError(f"存在未定义的字段: {', '.join(sorted(unknown))}")

        return cls(**data)

    # ===== 连接管理 =====
    @classmethod
    def configure_sources(
        cls,
        sources: Mapping[str, Union[Mapping[str, str], Any]],
        *,
        client_factory: Optional[Callable[[Dict[str, str]], "InfluxDBClient"]] = None,
        reset_cached_clients: bool = True,
    ) -> None:
        """
        注册数据源配置，如 {"sensor": {"url": "...", "token": "...", "org": "...", "bucket": "..."}}。
        client_factory 可自定义 InfluxDBClient 创建逻辑，默认直接用 influxdb_client.InfluxDBClient。
        """
        if not sources:
            raise ValueError("需要至少提供一个 influxdb 数据源配置。")

        normalized = {
            name: cls._normalize_source_conf(conf) for name, conf in sources.items()
        }
        cls._sources = normalized
        if reset_cached_clients:
            cls._clients.clear()
        if client_factory is not None:
            cls._client_factory = client_factory

    @staticmethod
    def _normalize_source_conf(conf: Union[Mapping[str, str], Any]) -> Dict[str, str]:
        if is_dataclass(conf):
            conf_dict = asdict(conf)
        elif isinstance(conf, Mapping):
            conf_dict = dict(conf)
        else:
            conf_dict = {
                k: getattr(conf, k)
                for k in ("url", "token", "org", "bucket")
                if hasattr(conf, k)
            }
        required = ("url", "token", "org", "bucket")
        missing = [k for k in required if not conf_dict.get(k)]
        if missing:
            raise ValueError(f"配置缺少必填项: {', '.join(missing)}")
        return {k: conf_dict[k] for k in required}

    @classmethod
    def _source_name(cls, source: Optional[str]) -> str:
        return source or getattr(cls, "db", None) or "default"

    @classmethod
    def _get_source_conf(
        cls, source: Optional[str] = None
    ) -> Tuple[str, Dict[str, str]]:
        if not cls._sources:
            raise RuntimeError(
                "InfluxDB2Base 未配置数据源，请先调用 configure_sources。"
            )
        name = cls._source_name(source)
        if name not in cls._sources:
            raise ValueError(f"未知的 influxdb 数据源: {name}")
        return name, cls._sources[name]

    @staticmethod
    def _default_client_factory(conf: Dict[str, str]) -> "InfluxDBClient":
        from influxdb_client import InfluxDBClient

        return InfluxDBClient(url=conf["url"], token=conf["token"], org=conf["org"])

    @classmethod
    def _get_client(
        cls, source: Optional[str] = None, conf: Optional[Dict[str, str]] = None
    ) -> "InfluxDBClient":
        name = cls._source_name(source)
        if conf is None:
            _, conf = cls._get_source_conf(name)
        if name not in cls._clients:
            factory = cls._client_factory or cls._default_client_factory
            cls._clients[name] = factory(conf)
        return cls._clients[name]

    @classmethod
    def _resolve_connection(
        cls,
        *,
        query_api: Any = None,
        bucket: Optional[str] = None,
        org: Optional[str] = None,
        source: Optional[str] = None,
    ) -> Tuple[Any, str, str]:
        """
        返回 (query_api, bucket, org)。已传入则透传，否则按数据源配置生成。
        """
        if query_api is not None and bucket is not None and org is not None:
            return query_api, bucket, org

        src_name, conf = cls._get_source_conf(source)
        bucket = bucket or conf["bucket"]
        org = org or conf["org"]
        if query_api is None:
            client = cls._get_client(src_name, conf)
            query_api = client.query_api()
        return query_api, bucket, org

    @classmethod
    def _resolve_write(
        cls,
        *,
        write_api: Any = None,
        bucket: Optional[str] = None,
        org: Optional[str] = None,
        source: Optional[str] = None,
    ) -> Tuple[Any, str, str]:
        """
        返回 (write_api, bucket, org)。已传入则透传，否则按数据源配置生成。
        """
        if write_api is not None and bucket is not None and org is not None:
            return write_api, bucket, org

        src_name, conf = cls._get_source_conf(source)
        bucket = bucket or conf["bucket"]
        org = org or conf["org"]
        if write_api is None:
            from influxdb_client.client.write_api import SYNCHRONOUS

            client = cls._get_client(src_name, conf)
            write_api = client.write_api(write_options=SYNCHRONOUS)
        return write_api, bucket, org

    @classmethod
    def _ensure_aware(
        cls, dt: datetime | None, tz: Union[str, ZoneInfo]
    ) -> datetime | None:
        return ensure_aware(dt, tz)

    @classmethod
    def _flux_time(cls, t: TimeLike) -> Optional[str]:
        return flux_time(t, cls.timezone)

    @staticmethod
    def _pandas_freq(freq: str) -> str:
        return pandas_freq(freq)

    @staticmethod
    def _flux_agg_fn(agg: Agg) -> str:
        return flux_agg_fn(agg)

    @staticmethod
    def _is_numeric_field(spec: FieldSpec) -> bool:
        dtype = spec.resolved_dtype()
        if dtype is None:
            return False
        if dtype is bool:
            return False
        if isinstance(dtype, type):
            try:
                return issubclass(dtype, numbers.Number)
            except TypeError:
                return False
        return False

    @staticmethod
    def _require_tags(spec: FieldSpec) -> Mapping[str, str]:
        if not spec.tags:
            raise ValueError(
                "FieldSpec.tags 不能为空，请至少提供一个 tag 键值对（如 {'name': '...'}）"
            )
        return spec.tags

    @classmethod
    def _pivot_tag_key(cls, spec_items: Sequence[Tuple[str, FieldSpec]]) -> str:
        tag_sets: List[set[str]] = []
        for _, spec in spec_items:
            tags = cls._require_tags(spec)
            tag_sets.append(set(tags.keys()))
        if not tag_sets:
            raise ValueError("未指定任何字段。")
        common_keys = set.intersection(*tag_sets)
        if "name" in common_keys:
            return "name"
        if common_keys:
            return sorted(common_keys)[0]
        raise ValueError(
            "各字段的 tags 需至少共享一个相同的 tag 键（如 name）以便 pivot。"
        )

    @staticmethod
    def _column_label(spec: FieldSpec, pivot_key: str, fallback: str) -> str:
        tags = spec.tags or {}
        if pivot_key in tags:
            return tags[pivot_key]
        if tags:
            return next(iter(tags.values()))
        return fallback

    @classmethod
    def _now(cls, tz_name: Optional[str] = None) -> datetime:
        tz = tz_name or cls.timezone
        if isinstance(tz, str):
            tz = ZoneInfo(tz)
        return datetime.now(tz)

    @staticmethod
    def _bool_from_scalar(value: Any) -> bool:
        if isinstance(value, bool):
            return value
        if isinstance(value, str):
            t = value.strip().lower()
            if t in {"true", "t", "1", "yes", "y", "on"}:
                return True
            if t in {"false", "f", "0", "no", "n", "off"}:
                return False
        try:
            return bool(float(value))
        except Exception as exc:
            raise ValueError(f"无法将值 {value!r} 转为 bool") from exc

    @classmethod
    def _convert_value_for_write(
        cls, value: Any, spec: FieldSpec, *, field_name: str
    ) -> Any:
        if value is None:
            return None
        try:
            if pd.isna(value):
                return None
        except Exception:
            # 非标量类型（如 dict）不适用 pd.isna，忽略
            pass

        dtype = spec.resolved_dtype()
        if dtype is None:
            return value

        if callable(dtype) and not isinstance(dtype, type):
            try:
                return dtype(value)
            except Exception as exc:
                raise ValueError(
                    f"{field_name}: 无法将值 {value!r} 转换为 {dtype}"
                ) from exc

        try:
            if dtype is bool:
                return cls._bool_from_scalar(value)
            if dtype is int:
                return int(value)
            if dtype is float:
                return float(value)
            if dtype is str:
                if isinstance(value, datetime):
                    value = value.isoformat()
                return str(value)
            if dtype in (datetime, pd.Timestamp):
                if isinstance(value, pd.Timestamp):
                    return value.to_pydatetime()
                if isinstance(value, datetime):
                    return value
                parsed = pd.to_datetime(value, errors="raise")
                return (
                    parsed.to_pydatetime()
                    if isinstance(parsed, pd.Timestamp)
                    else parsed
                )
            return dtype(value)
        except Exception as exc:
            raise ValueError(
                f"{field_name}: 无法将值 {value!r} 转换为 {dtype}"
            ) from exc

    # ===== 写入辅助 =====
    @classmethod
    def _normalize_time_for_write(
        cls, t: Any, tz: Optional[Union[str, ZoneInfo]]
    ) -> Any:
        if t is None:
            return None
        if isinstance(t, datetime):
            tz = tz or cls.timezone
            return cls._ensure_aware(t, tz)
        return t  # 其他类型（如 str）直接透传给 Influx

    @classmethod
    def _build_points(
        cls,
        rows: Sequence[Union[Mapping[str, Any], Any]],
        *,
        fields: Optional[Sequence[str]] = None,
        time_key: str = "_time",
        tz_name: Optional[str] = None,
        default_time: Any = None,
        strict_keys: bool = True,
    ) -> List[Any]:
        """
        将行数据（Mapping）转换为 Point 列表。行里的 key 用字段名（类属性名），时间用 time_key。
        """
        from influxdb_client import Point

        if isinstance(rows, Mapping):
            rows = [rows]  # 单条也可写

        spec_items = cls._specs_from_fields(fields)
        if not spec_items:
            raise ValueError("未指定任何字段。")

        points: List[Any] = []
        tz_name = tz_name or cls.timezone
        default_ts = cls._normalize_time_for_write(default_time, tz_name)
        allowed_fields = {col_name for col_name, _ in spec_items}
        allowed_time_keys = {time_key, "time"}

        for row in rows:
            if not isinstance(row, Mapping):
                raise TypeError("写入数据必须是 Mapping 或其列表。")

            if strict_keys:
                unknown = set(row.keys()) - allowed_fields - allowed_time_keys
                if unknown:
                    raise ValueError(f"存在未定义的字段: {', '.join(sorted(unknown))}")

            ts_raw = row.get(time_key)
            if ts_raw is None and "time" in row:
                ts_raw = row.get("time")
            if ts_raw is None:
                ts_raw = default_ts
            ts = cls._normalize_time_for_write(ts_raw, tz_name)
            for col_name, spec in spec_items:
                if col_name not in row:
                    continue
                val = cls._convert_value_for_write(
                    row[col_name], spec, field_name=col_name
                )
                if val is None:
                    continue
                tags = cls._require_tags(spec)
                p = Point(cls.measurement)
                for tag_key, tag_val in tags.items():
                    p = p.tag(tag_key, tag_val)
                p = p.field(spec.field, val)
                if ts is not None:
                    p = p.time(ts)
                points.append(p)

        return points

    # ===== 实例写入 =====
    def _resolve_write_time(
        self, *, time: Any = None, tz_name: Optional[str] = None
    ) -> Any:
        cls = self.__class__
        tz = tz_name or cls.timezone
        resolved = cls._normalize_time_for_write(
            time if time is not None else self._values.get("time"),
            tz,
        )
        if resolved is None:
            resolved = cls._now(tz_name or cls.timezone)
        return resolved

    def _points_from_values(
        self,
        *,
        tz_name: Optional[str] = None,
        time: Any = None,
        fields: Optional[Sequence[str]] = None,
        strict_keys: bool = True,
    ) -> Tuple[List[Any], Any]:
        cls = self.__class__
        resolved_time = self._resolve_write_time(time=time, tz_name=tz_name)
        row = {**self._values, "time": resolved_time}
        points = cls._build_points(
            row,
            fields=fields,
            time_key="time",
            tz_name=tz_name,
            default_time=resolved_time,
            strict_keys=strict_keys,
        )
        return points, resolved_time

    def to_line_protocol(
        self,
        *,
        tz_name: Optional[str] = None,
        time: Any = None,
        fields: Optional[Sequence[str]] = None,
        precision: str = "s",
    ) -> str:
        """
        把当前实例转换为 line protocol 字符串，便于调试或直接写入。
        若未指定时间则使用类定义的 _now 默认时间。
        """
        points, resolved_time = self._points_from_values(
            tz_name=tz_name,
            time=time,
            fields=fields,
        )
        if not points:
            raise ValueError("未提供任何可写入的字段值。")
        lp = "\n".join(p.to_line_protocol(precision=precision) for p in points)
        # 把实际写入的时间回写到实例，方便后续查看
        self._values["time"] = resolved_time
        return lp

    def save(
        self,
        *,
        precision: str = "s",
        tz_name: Optional[str] = None,
        source: Optional[str] = None,
        write_api: Any = None,
        bucket: Optional[str] = None,
        org: Optional[str] = None,
        time: Any = None,
        fields: Optional[Sequence[str]] = None,
    ) -> str:
        """
        保存当前实例到 InfluxDB。返回写入使用的 line protocol 字符串。
        默认使用 s 精度以匹配 Point.to_line_protocol 的时间戳格式。
        """
        cls = self.__class__
        if getattr(cls, "readonly", False):
            raise RuntimeError(f"{cls.__name__} 为只读模型，禁止写入。")
        lp = self.to_line_protocol(
            tz_name=tz_name,
            time=time,
            fields=fields,
            precision=precision,
        )
        write_api, bucket, org = cls._resolve_write(
            write_api=write_api, bucket=bucket, org=org, source=source
        )
        write_api.write(bucket=bucket, org=org, record=lp, write_precision=precision)
        return lp

    # ===== 字段规格 =====
    @classmethod
    def columns(cls) -> List[str]:
        return [k for k, v in cls.__dict__.items() if isinstance(v, FieldSpec)]

    @classmethod
    def _specs_from_fields(
        cls, fields: Optional[Sequence[str]]
    ) -> List[Tuple[str, FieldSpec]]:
        all_items = [
            (k, v) for k, v in cls.__dict__.items() if isinstance(v, FieldSpec)
        ]
        if not fields:
            return all_items
        wanted = set(fields)
        out = [(k, v) for k, v in all_items if k in wanted]
        missing = wanted - {k for k, _ in out}
        if missing:
            raise ValueError(f"未知的字段: {', '.join(sorted(missing))}")
        return out

    @classmethod
    def _property_fields(cls) -> List[str]:
        return [k for k, v in cls.__dict__.items() if isinstance(v, property)]

    @classmethod
    def _normalize_record(
        cls,
        rec: Mapping[str, Any],
        allowed_fields: set[str],
        tag_value_to_field: Dict[str, str],
    ) -> Dict[str, Any]:
        normalized: Dict[str, Any] = {}
        for key, val in rec.items():
            if key in {"_time", "time"}:
                normalized["time"] = val
                continue
            if key in allowed_fields:
                normalized[key] = val
                continue
            if key in tag_value_to_field:
                normalized[tag_value_to_field[key]] = val
        return normalized

    @classmethod
    def _fast_instance(cls: Type[TInflux], normalized: Mapping[str, Any]) -> TInflux:
        obj = cls.__new__(cls)  # type: ignore[misc]
        obj._values = dict(normalized)
        return obj  # type: ignore[return-value]

    # ===== Flux 查询生成 =====
    @classmethod
    def _get_query_str(
        cls,
        *,
        bucket: str,
        start_time: TimeLike,
        end_time: TimeLike,
        freq: str = "1s",
        fields: Optional[Sequence[str]] = None,
        create_empty_in_flux: bool = False,
    ) -> str:
        spec_items = cls._specs_from_fields(fields)
        if not spec_items:
            raise ValueError("未指定任何字段。")
        pivot_key = cls._pivot_tag_key(spec_items)

        flux_start = cls._flux_time(start_time)
        flux_stop = cls._flux_time(end_time)

        range_line = (
            f"|> range(start: {flux_start})\n"
            if flux_stop is None
            else f"|> range(start: {flux_start}, stop: {flux_stop})\n"
        )

        # 按 agg 分组
        agg_map: Dict[Agg, List[FieldSpec]] = {}
        for _, spec in spec_items:
            agg_map.setdefault(spec.agg, []).append(spec)

        parts: List[str] = []
        tnames: List[str] = []

        for i, (agg, specs) in enumerate(agg_map.items()):
            tname = f"p{i}"
            tnames.append(tname)

            field_filters = []
            for spec in specs:
                tags = cls._require_tags(spec)
                tag_filters = [f'r["{k}"]=="{v}"' for k, v in tags.items()]
                tag_filters.append(f'r._field=="{spec.field}"')
                field_filters.append("(" + " and ".join(tag_filters) + ")")
            field_filters_str = " or ".join(field_filters)

            flux_fn = cls._flux_agg_fn(agg)
            create_empty = "true" if create_empty_in_flux else "false"
            keep_cols = f'"_time","{pivot_key}","_field","_value"'

            s = (
                f"{tname} =\n"
                f'  from(bucket:"{bucket}")\n'
                f"{range_line}"
                f'    |> filter(fn:(r)=> r._measurement=="{cls.measurement}")\n'
                f"    |> filter(fn:(r)=> {field_filters_str})\n"
                f"    |> aggregateWindow(every: {freq}, fn: {flux_fn}, createEmpty: {create_empty})\n"
                f"    |> keep(columns: [{keep_cols}])\n"
            )
            parts.append(s)

        query = "\n\n".join(parts)
        if len(tnames) == 1:
            # 只有一种聚合时直接使用单表，避免 union 报 “至少需要两个输入流”
            query += f"\n\n{tnames[0]}\n"
        else:
            query += f"\n\nunion(tables:[{', '.join(tnames)}])\n"
        query += f'|> group(columns: ["_field","{pivot_key}"])\n'
        query += f'|> pivot(rowKey: ["_time"], columnKey: ["{pivot_key}"], valueColumn: "_value")\n'
        return query

    @staticmethod
    def _attach_query_to_exc(exc: BaseException, query: str) -> None:
        """把生成的 Flux 语句附加到异常上，便于排查。"""
        try:
            setattr(exc, "query", query)
        except Exception:
            pass
        try:
            exc.add_note(f"Flux query:\n{query}")
        except Exception:
            pass

    # ===== DataFrame 合并 + 补点 =====
    @classmethod
    def _concat_dfs(cls, dfs: pd.DataFrame | List[pd.DataFrame]) -> pd.DataFrame:
        def _clean(d: pd.DataFrame) -> pd.DataFrame:
            if d is None or d.empty or "_time" not in d.columns:
                return pd.DataFrame()
            d = d.drop(
                columns=["result", "table", "_start", "_stop", "_field"],
                errors="ignore",
            )
            d = d.dropna(subset=["_time"])
            d = d.set_index("_time").sort_index()
            return d

        if dfs is None:
            return pd.DataFrame()

        if isinstance(dfs, list):
            if not dfs:
                return pd.DataFrame()
            cleaned = [_clean(d) for d in dfs]
            cleaned = [d for d in cleaned if not d.empty]
            if not cleaned:
                return pd.DataFrame()
            df = pd.concat(cleaned, axis=1).reset_index()
        else:
            df = _clean(dfs)
            if df.empty:
                return pd.DataFrame()
            df = df.reset_index()
        return df

    @classmethod
    def _gapfill(
        cls,
        df: pd.DataFrame,
        *,
        start_time: TimeLike,
        end_time: TimeLike,
        freq: str,
        fields: Optional[Sequence[str]] = None,
        tz_name: Optional[str] = None,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = False,
    ) -> pd.DataFrame:
        """
        pandas 补点：生成完整时间索引并按 FieldSpec.fill 分列填充。
        - Fill.none: 不动
        - Fill.last: ffill（可选 fill_initial=True 再 bfill）
        - Fill.interpolate: 数值列按时间插值（边界可选 ffill/bfill）
        """
        spec_items = cls._specs_from_fields(fields)
        pivot_key = cls._pivot_tag_key(spec_items) if spec_items else "name"
        colspec: Dict[str, FieldSpec] = {
            col_name: spec for col_name, spec in spec_items
        }
        tag_value_to_col = {
            cls._column_label(spec, pivot_key, col_name): col_name
            for col_name, spec in spec_items
        }

        if df is None or df.empty:
            # 返回一个只包含 _time 的空表（仍可按输入时间生成索引）
            return pd.DataFrame(columns=["_time"] + list(colspec.keys()))

        # 时间列标准化
        tz_name = tz_name or cls.timezone
        t = pd.to_datetime(df["_time"], errors="coerce")
        if getattr(t.dt, "tz", None) is None:
            t = t.dt.tz_localize("UTC")
        t = t.dt.tz_convert(tz_name)
        df = df.copy()
        df["_time"] = t

        # 可选：把点位列名重命名为字段名（ycsyl/fsf/...）
        if rename_columns_to_fieldname:
            df = df.rename(columns=tag_value_to_col)

        # 建索引、去重
        df = (
            df.sort_values("_time")
            .drop_duplicates(subset=["_time"], keep="last")
            .set_index("_time")
        )
        # 补点范围：能解析为 datetime 就用它，否则退化用 df 边界
        s_abs = (
            cls._ensure_aware(start_time, tz_name)
            if isinstance(start_time, datetime)
            else None
        )
        e_abs = (
            cls._ensure_aware(end_time, tz_name)
            if isinstance(end_time, datetime)
            else None
        )
        if s_abs is not None:
            s_abs = (
                pd.Timestamp(s_abs).tz_convert(tz_name)
                if pd.Timestamp(s_abs).tzinfo
                else pd.Timestamp(s_abs).tz_localize(tz_name)
            )
        if e_abs is not None:
            e_abs = (
                pd.Timestamp(e_abs).tz_convert(tz_name)
                if pd.Timestamp(e_abs).tzinfo
                else pd.Timestamp(e_abs).tz_localize(tz_name)
            )

        start_idx = (s_abs if s_abs is not None else df.index.min()).floor("s")
        if e_abs is not None:
            end_idx = e_abs.floor("s")
        else:
            end_idx = df.index.max().ceil("s")
        if end_idx < start_idx:
            end_idx = start_idx

        pd_freq = cls._pandas_freq(freq)
        full_idx = pd.date_range(
            start=start_idx, end=end_idx, freq=pd_freq, tz=start_idx.tz
        )

        df = df.reindex(full_idx)

        # 逐列填充
        for col_name, spec in colspec.items():
            if col_name not in df.columns:
                continue

            is_numeric = cls._is_numeric_field(spec)
            if is_numeric:
                df[col_name] = pd.to_numeric(df[col_name], errors="coerce")

            if spec.fill == Fill.none:
                continue

            if spec.fill == Fill.last:
                df[col_name] = df[col_name].ffill()
                if fill_initial:
                    df[col_name] = df[col_name].bfill()
                continue

            if spec.fill == Fill.interpolate:
                if not is_numeric:
                    # 非数值列不插值
                    continue
                df[col_name] = df[col_name].interpolate(method="time")
                # 插值的边界处理（可选）
                df[col_name] = df[col_name].ffill()
                if fill_initial:
                    df[col_name] = df[col_name].bfill()
                continue

        return df.reset_index(names="_time")

    # ===== 对外接口 =====
    @classmethod
    def _read_last_df(
        cls: Type[TInflux],
        *,
        start_time: TimeLike,
        end_time: TimeLike = None,
        freq: str = "5s",
        fields: Optional[Sequence[str]] = None,
        create_empty_in_flux: bool = True,
        tz_name: Optional[str] = None,
        do_gapfill: bool = True,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = True,
        now: Optional[datetime] = None,
        tolerance: Optional[float | int] = None,
        include_properties: bool = False,
        query_api=None,
        org: Optional[str] = None,
        bucket: Optional[str] = None,
        source: Optional[str] = None,
    ) -> pd.DataFrame:
        now_dt = cls._ensure_aware(now, tz_name or cls.timezone) or cls._now(tz_name)
        resolved_start = cls._resolve_time_like(start_time, now_dt)
        resolved_end = cls._resolve_time_like(end_time, now_dt)
        if resolved_start is None:
            raise ValueError("start_time must be specified")
        if resolved_end is None:
            resolved_end = now_dt
        elif isinstance(resolved_end, datetime) and resolved_end > now_dt:
            resolved_end = now_dt
        query_api, bucket, org = cls._resolve_connection(
            query_api=query_api, bucket=bucket, org=org, source=source
        )
        query: Optional[str] = None
        try:
            query = cls._get_query_str(
                bucket=bucket,
                start_time=resolved_start,
                end_time=resolved_end,
                freq=freq,
                fields=fields,
                create_empty_in_flux=create_empty_in_flux,
            )
            dfs = query_api.query_data_frame(org=org, query=query)
            df = cls._concat_dfs(dfs)

            if do_gapfill:
                df = cls._gapfill(
                    df,
                    start_time=resolved_start,
                    end_time=resolved_end,
                    freq=freq,
                    fields=fields,
                    tz_name=tz_name,
                    rename_columns_to_fieldname=rename_columns_to_fieldname,
                    fill_initial=fill_initial,
                )
            df = cls._cast_types(
                df,
                fields=fields,
                rename_columns_to_fieldname=rename_columns_to_fieldname,
                tz_name=tz_name or cls.timezone,
            )
            if df is None:
                df = pd.DataFrame()

            if include_properties:
                props = cls._property_fields()
                if props:
                    spec_items = cls._specs_from_fields(fields)
                    pivot_key = cls._pivot_tag_key(spec_items)
                    tag_value_to_field = {
                        cls._column_label(spec, pivot_key, col_name): col_name
                        for col_name, spec in spec_items
                    }
                    allowed_fields = {col_name for col_name, _ in spec_items}
                    df = df.copy()
                    for prop in props:
                        df[prop] = df.apply(
                            lambda row, p=prop: getattr(
                                cls._fast_instance(
                                    cls._normalize_record(
                                        row, allowed_fields, tag_value_to_field
                                    )
                                ),
                                p,
                            ),
                            axis=1,
                        )

            # 统一对外时间列为 time，移除内部 _time
            if "_time" in df.columns:
                if "time" not in df.columns:
                    df = df.rename(columns={"_time": "time"})
                else:
                    df = df.drop(columns=["_time"])

            if tolerance is not None and df is not None and not df.empty:
                ts = pd.to_datetime(df["time"], errors="coerce")
                if getattr(ts.dt, "tz", None) is None:
                    ts = ts.dt.tz_localize(tz_name or cls.timezone)
                last_ts = ts.iloc[-1]
                if last_ts is not pd.NaT:
                    delta = (now_dt - last_ts).total_seconds()
                    if delta > float(tolerance):
                        raise Exception("最新数据与当前时间差异大于容忍度")
            df.attrs["query"] = query
            return df
        except Exception as exc:
            if query:
                cls._attach_query_to_exc(exc, query)
            raise

    @classmethod
    def read_last(
        cls: Type[TInflux],
        *,
        start_time: TimeLike,
        end_time: TimeLike = None,
        freq: str = "5s",
        fields: Optional[Sequence[str]] = None,
        create_empty_in_flux: bool = True,
        tz_name: Optional[str] = None,
        do_gapfill: bool = True,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = True,
        now: Optional[datetime] = None,
        tolerance: Optional[float | int] = None,
        query_api=None,
        org: Optional[str] = None,
        bucket: Optional[str] = None,
        source: Optional[str] = None,
        lazy: bool = True,
    ) -> InfluxQuerySet[TInflux]:
        """
        返回延迟实例化的类实例序列。
        """
        spec_items = cls._specs_from_fields(fields)
        pivot_key = cls._pivot_tag_key(spec_items) if spec_items else "name"
        tag_value_to_field = {
            cls._column_label(spec, pivot_key, col_name): col_name
            for col_name, spec in spec_items
        }
        allowed_fields = {col_name for col_name, _ in spec_items}
        allowed_fields |= set(cls._property_fields())

        df = cls._read_last_df(
            start_time=start_time,
            end_time=end_time,
            freq=freq,
            fields=fields,
            create_empty_in_flux=create_empty_in_flux,
            tz_name=tz_name,
            do_gapfill=do_gapfill,
            rename_columns_to_fieldname=rename_columns_to_fieldname,
            fill_initial=fill_initial,
            now=now,
            tolerance=tolerance,
            include_properties=False,
            query_api=query_api,
            org=org,
            bucket=bucket,
            source=source,
        )
        if df is None:
            df = pd.DataFrame()

        query_str = getattr(df, "attrs", {}).get("query")
        lazy_seq = InfluxQuerySet(
            df, cls, allowed_fields, tag_value_to_field, query=query_str
        )
        if df.empty:
            return lazy_seq

        if not lazy:
            cache = list(
                df.apply(
                    lambda row: cls._fast_instance(
                        cls._normalize_record(row, allowed_fields, tag_value_to_field)
                    ),
                    axis=1,
                )
            )
            lazy_seq._cache = cache  # type: ignore[attr-defined]
        return lazy_seq

    @classmethod
    def read_last_dataframe(
        cls,
        *,
        start_time: TimeLike,
        end_time: TimeLike = None,
        freq: str = "5s",
        fields: Optional[Sequence[str]] = None,
        create_empty_in_flux: bool = True,
        tz_name: Optional[str] = None,
        do_gapfill: bool = True,
        rename_columns_to_fieldname: bool = True,
        fill_initial: bool = True,
        now: Optional[datetime] = None,
        tolerance: Optional[float | int] = None,
        query_api=None,
        org: Optional[str] = None,
        bucket: Optional[str] = None,
        source: Optional[str] = None,
    ) -> pd.DataFrame:
        """返回 DataFrame 结果，避免实例化开销。"""
        return cls._read_last_df(
            start_time=start_time,
            end_time=end_time,
            freq=freq,
            fields=fields,
            create_empty_in_flux=create_empty_in_flux,
            tz_name=tz_name,
            do_gapfill=do_gapfill,
            rename_columns_to_fieldname=rename_columns_to_fieldname,
            fill_initial=fill_initial,
            now=now,
            tolerance=tolerance,
            include_properties=True,
            query_api=query_api,
            org=org,
            bucket=bucket,
            source=source,
        )

    @classmethod
    def _resolve_time_like(cls, t: TimeLike, now: datetime) -> TimeLike:
        return resolve_time_like(t, now, cls.timezone)

    @classmethod
    def _cast_types(
        cls,
        df: pd.DataFrame,
        *,
        fields: Optional[Sequence[str]],
        rename_columns_to_fieldname: bool,
        tz_name: Optional[str] = None,
    ) -> pd.DataFrame:
        if df is None or df.empty:
            return df

        spec_items = cls._specs_from_fields(fields)
        pivot_key = cls._pivot_tag_key(spec_items) if spec_items else "name"
        col_map: Dict[str, FieldSpec] = {}
        for col_name, spec in spec_items:
            key = (
                col_name
                if rename_columns_to_fieldname
                else cls._column_label(spec, pivot_key, col_name)
            )
            col_map[key] = spec

        df = df.copy()
        if "time" in df.columns and "_time" not in df.columns:
            df["_time"] = df["time"]
        post_funcs: Dict[str, Callable[[Mapping[str, Any]], Any]] = {}
        for col, spec in col_map.items():
            if col not in df.columns:
                continue
            target = spec.target_type()
            if target is None:
                target = spec.dtype
            df[col] = cast_series(
                df[col],
                target,
                tz_name=tz_name,
                src_tz=spec.src_tz or tz_name,
            )
            if spec.post is not None:
                post_funcs[col] = spec.post

        # 行级后处理，可使用整行值进行计算
        for col, fn in post_funcs.items():
            df[col] = df.apply(lambda row: fn(row), axis=1)
        return df
